##########################################################
####### Single Starter/Generator electrical system #######
####################    Syd Adams    #####################
###### Based on Curtis Olson's nasal electrical code #####
##########################################################
##########################################################
######## Modified by Clement DE L'HAMAIDE for P92 ########
##########################################################
######## Modified by Bea Wolf			  ########
######## for CH650 2020-2023				  ########
##########################################################

var OutPuts = props.globals.getNode("/systems/electrical/outputs",1);
var Volts = props.globals.getNode("/systems/electrical/volts",1);
var Amps = props.globals.getNode("/systems/electrical/amps",1);
var delta_t = props.globals.getNode("/sim/time/delta-sec");
var eng_amp = props.globals.initNode("/engines/engine/amp-v", 0.0, "DOUBLE");

var sw = props.globals.getNode("/controls/switches");
var sw_e = props.globals.getNode("/controls/electric");
var sw_l = props.globals.getNode("/controls/lighting");
var cb = props.globals.initNode("/controls/circuit-breakers");

var serviceable = {
	electric:	props.globals.getNode("systems/electrical/serviceable",1),
	comm:		props.globals.getNode("/instrumentation/comm/serviceable"),
	xpdr:		props.globals.getNode("instrumentation/transponder/serviceable"),
};
var switches = {
	bat:			props.globals.getNode("/controls/engines/engine/master-bat",1),
	alt:			props.globals.getNode("/controls/engines/engine/master-alt",1),
	radio_master:	sw_e.initNode("radio-master",	1,	"BOOL"),
	starter:		props.globals.getNode("/controls/engines/engine[0]/starter", 1),
	strobe:		sw_l.initNode("strobe",		0,	"BOOL"),
	nav:			sw_l.initNode("nav-lights",	0,	"BOOL"),
	landing:		sw_l.initNode("landing-light",	0,	"BOOL"),
	taxi:			sw_l.initNode("taxi-light",	0,	"BOOL"),
	boost:		props.globals.initNode("/controls/engines/engine/fuel-pump", 0, "BOOL"),
};
var circuit_breakers = {
	com:		cb.initNode("com",	1,	"BOOL"),
	nav:		cb.initNode("nav",	1,	"BOOL"),
	xpdr:		cb.initNode("xpdr",	1,	"BOOL"),
	audio:	cb.initNode("audio",	1,	"BOOL"),
	gps:		cb.initNode("gps",	1,	"BOOL"),
	efis:		cb.initNode("efis",	1,	"BOOL"),
	ens:		cb.initNode("ens",	1,	"BOOL"),
	trim:		cb.initNode("trim",	1,	"BOOL"),
	spare:	cb.initNode("spare",	1,	"BOOL"),
	start:	cb.initNode("start",	1,	"BOOL"),
	field:	cb.initNode("field",	1,	"BOOL"),
	flap:		cb.initNode("flap",	1,	"BOOL"),
	stall:	cb.initNode("stall",	1,	"BOOL"),
	aux:		cb.initNode("aux",	1,	"BOOL"),
	alt:		cb.initNode("alt",	1,	"BOOL"),
	batt:		cb.initNode("batt",	1,	"BOOL"),
};

# Estimate electrical load:
# Landing Light: 	(based on https://aeroleds.com/products/sunspot-36lx-landing/): 45 Watts
# Taxi Light:	(based on https://aeroleds.com/products/sunspot-36lx-taxi/): 45 Watts
# Navigation Lights and Strobe Lights based on https://aeroleds.com/wp-content/uploads/2018/02/0008-0006-Linear-Pulsar-NS-Series-Installation.pdf:
#	Navigation:		8.4 Watts (steady)			-> *2
#	Strobe:		70 Watts (during flash, 0.2 secs)	-> *2
#				11.2 Watts (average)			-> *2
# 	Fuel Pump: 		50 Watts (complete guess)
var consumers_main = { 
	# consumer name:  [  power (watts),  switch node,   circuit breaker node, output node ],
	landing_light:	[ 45, 	switches.landing, nil,					OutPuts.initNode("landing-light", 0.0, "DOUBLE")	],
	taxi_light:		[ 45, 	switches.taxi,    nil,					OutPuts.initNode("taxi-light", 0.0, "DOUBLE")		],
	strobe_lights:	[ 22.4, 	switches.strobe, 	nil,					OutPuts.initNode("strobe-lights", 0.0, "DOUBLE")	],
	nav_lights:		[ 16.8, 	switches.nav,   	nil,					OutPuts.initNode("nav-lights", 0.0, "DOUBLE")		],
	fuel_pump:		[ 50.0, 	switches.boost, 	nil,					OutPuts.initNode("fuel-pump", 0.0, "DOUBLE")		],
	# Starter has about 900W power according to https://www.ulforum.de/ultraleicht/forum/2_technik-und-flugzeuge/2466_anlasser-rotax-912-uls/seite-4.html
	starter:		[ 900.0,	switches.starter, circuit_breakers.start,		OutPuts.initNode("starter", 0.0, "DOUBLE")	],
	
	trim:			[ 1.0,	nil,			circuit_breakers.trim,		OutPuts.initNode("trim",    0.0, "DOUBLE")		],
};

var avionics = {
	# consumer name:  [  power (watts),  switch node,   circuit breaker node, output node ],
	# Apollo SL40: 4W (receiving), 28W (transmit) ref. http://rv7-a.com/manuals/SL40Comm_InstallationManual.pdf p. 15 (printed page number)
	comm:			[ 4, serviceable.comm, circuit_breakers.com,	OutPuts.initNode("comm", 0.0, "DOUBLE") ],
	# Garmin GTX-327: 15W (typical), 22W (max) ref. http://www.expaircraft.com/PDF/GTX327-IM.pdf p. 2 (printed page number)
	transponder:	[ 15, serviceable.xpdr, circuit_breakers.xpdr,	OutPuts.initNode("transponder", 0.0, "DOUBLE") ],
	# Dynon EFIS D-180: 14W (typical), 19W (max) ref. https://dynonavionics.com/includes/guides/FlightDEK-D180_Installation_Guide_Rev_H.pdf p. 9-38 (printed page number)
	efis:			[ 14, nil, circuit_breakers.efis, OutPuts.initNode("efis", 0.0, "DOUBLE") ],
};

var strobe = aircraft.light.new("/sim/model/lights/strobe-lights", [0.2, 1.25], "/systems/electrical/outputs/strobe-lights");

#var battery = Battery.new(volts,amps,amp_hours,charge_norm,charge_amps);

var Battery = {
	new : func( ideal_volts, ideal_amps, amp_hours, charge_norm, charge_amps ) {
		m = { parents : [Battery] };
		m.ideal_volts = ideal_volts;
		m.ideal_amps = ideal_amps;
		m.amp_hours = amp_hours;
		m.charge_norm = charge_norm;
		m.charge_amps = charge_amps;
		return m;
	},
	apply_load : func {
		var amphrs_used = arg[0] * (arg[1] / 3600.0);
		var percent_used = amphrs_used / me.amp_hours;
		me.charge_norm -= percent_used;
		if ( me.charge_norm < 0.0 ) {
			me.charge_norm = 0.0;
		} elsif ( me.charge_norm > 1.0 ) {
			me.charge_norm = 1.0;
		}
		return me.amp_hours * me.charge_norm;
	},
	get_output_volts : func {
		var x = 1.0 - me.charge_norm;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_volts * factor;
	},
	get_output_amps : func {
		var x = 1.0 - me.charge_norm;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_amps * factor;
	},	
	recharge : func {
		me.charge_norm = 1.0;
	},
};

# var alternator = Alternator.new("rpm-source",rpm_threshold,volts,amps);

var Alternator = {
	new : func( rpm_source, rpm_threshold, ideal_volts, ideal_amps ) {
		m = { parents : [Alternator] };
		m.rpm_source =  props.globals.getNode( rpm_source,1);
		m.rpm_threshold = rpm_threshold;
		m.ideal_volts = ideal_volts;
		m.ideal_amps = ideal_amps;
		return m;
	},
	
	apply_load : func( amps, dt) {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ){
			factor = 1.0;
		}
		var available_amps = me.ideal_amps * factor;
		return available_amps - amps;
	},
	
	get_output_volts : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		return me.ideal_volts * factor;
	},
	
	get_output_amps : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		return me.ideal_amps * factor;
	}
};

var battery = Battery.new(12.0, 20.0, 22.0, 1.0, 12.0);
var alternator = Alternator.new("/engines/engine[0]/rpm", 250.0, 14.0, 40.0);

var update_virtual_bus = func {
	var dt = delta_t.getDoubleValue();
	var AltVolts = alternator.get_output_volts();
	var AltAmps = alternator.get_output_amps();
	var BatVolts = battery.get_output_volts();
	var BatAmps = battery.get_output_amps();
	var power_source = nil;
	var bus_volts = 0.0;
	var load = 0.0;
	
	if( serviceable.electric.getBoolValue() ){
		if ( switches.alt.getBoolValue() and circuit_breakers.alt.getBoolValue() and (AltAmps > BatAmps)){
			bus_volts = AltVolts;
			power_source = "alternator";
		}elsif ( switches.bat.getBoolValue() and circuit_breakers.batt.getBoolValue() ){
			bus_volts = BatVolts;
			power_source = "battery";
		}
	}
	
	load = 0.0;
	load += electrical_bus(bus_volts);
	load += avionics_bus(bus_volts);
	
	eng_amp.setDoubleValue(bus_volts);		# TODO what does this affect?
	
	var bus_amps = 0.0;
	
	if (bus_volts > 1.0){
		if (power_source == "battery"){
			bus_amps = BatAmps - load;
		} else {
			bus_amps = battery.charge_amps;
		}
	}
	
	if( power_source == "alternator" ){
		battery.apply_load(-battery.charge_amps, dt);
	}elsif( power_source == "battery" ){
		battery.apply_load(load, dt);
	} elsif( load != 0.0 ){
		die("ELECTRICAL: Current leakage!");
	}
		
	
	Amps.setValue(bus_amps);
	Volts.setValue(bus_volts);
	return load;
}

var electrical_bus = func(bus_volts){
	load = 0.0;
	
	foreach( var key; keys( consumers_main ) ){
		if( ( consumers_main[key][1] == nil or consumers_main[key][1].getBoolValue() ) and ( consumers_main[key][2] == nil or consumers_main[key][2].getBoolValue() ) ){
			consumers_main[key][3].setDoubleValue( bus_volts );
			if( bus_volts != 0 ){
				load += ( consumers_main[key][0] / bus_volts );
			}
		} else {
			consumers_main[key][3].setDoubleValue( 0.0 );
		}
	}
	
	return load;
}

var avionics_bus = func(bv) {
	load = 0.0;
	var bus_volts = 0.0;
	
	if( switches.radio_master.getBoolValue() ){
		var bus_volts = bv;
	}
		
	foreach( var key; keys( avionics ) ){
		if( ( avionics[key][1] == nil or avionics[key][1].getBoolValue() ) and ( avionics[key][2] == nil or avionics[key][2].getBoolValue() ) ){
			avionics[key][3].setDoubleValue( bus_volts );
			if( bus_volts != 0 ){
				load += ( avionics[key][0] / bus_volts );
			}
		} else {
			avionics[key][3].setDoubleValue( 0.0 );
		}
	}	
	
	return load;
}

var electrical_updater = maketimer( 0.0, update_virtual_bus );
electrical_updater.simulatedTime = 1;

setlistener("/sim/signals/fdm-initialized", func {
	electrical_updater.start();
});
