/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.processor;

import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.camel.AsyncCallback;
import org.apache.camel.CamelContext;
import org.apache.camel.Exchange;
import org.apache.camel.Expression;
import org.apache.camel.Processor;
import org.apache.camel.RuntimeExchangeException;
import org.apache.camel.Traceable;
import org.apache.camel.processor.DelegateAsyncProcessor;
import org.apache.camel.processor.ThrottlerRejectedExecutionException;
import org.apache.camel.spi.IdAware;
import org.apache.camel.util.AsyncProcessorHelper;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Throttler
extends DelegateAsyncProcessor
implements Traceable,
IdAware {
    private static final String PROPERTY_EXCHANGE_QUEUED_TIMESTAMP = "CamelThrottlerExchangeQueuedTimestamp";
    private static final String PROPERTY_EXCHANGE_STATE = "CamelThrottlerExchangeState";
    private final Logger log = LoggerFactory.getLogger(Throttler.class);
    private final CamelContext camelContext;
    private final DelayQueue<ThrottlePermit> delayQueue = new DelayQueue();
    private final ExecutorService asyncExecutor;
    private final boolean shutdownAsyncExecutor;
    private volatile long timePeriodMillis;
    private volatile int throttleRate;
    private String id;
    private Expression maxRequestsPerPeriodExpression;
    private boolean rejectExecution;
    private boolean asyncDelayed;
    private boolean callerRunsWhenRejected = true;

    public Throttler(CamelContext camelContext, Processor processor, Expression maxRequestsPerPeriodExpression, long timePeriodMillis, ExecutorService asyncExecutor, boolean shutdownAsyncExecutor, boolean rejectExecution) {
        super(processor);
        this.camelContext = camelContext;
        this.rejectExecution = rejectExecution;
        this.shutdownAsyncExecutor = shutdownAsyncExecutor;
        ObjectHelper.notNull(maxRequestsPerPeriodExpression, "maxRequestsPerPeriodExpression");
        this.maxRequestsPerPeriodExpression = maxRequestsPerPeriodExpression;
        if (timePeriodMillis <= 0L) {
            throw new IllegalArgumentException("TimePeriodMillis should be a positive number, was: " + timePeriodMillis);
        }
        this.timePeriodMillis = timePeriodMillis;
        this.asyncExecutor = asyncExecutor;
    }

    @Override
    public boolean process(Exchange exchange, AsyncCallback callback) {
        long queuedStart = 0L;
        if (this.log.isTraceEnabled()) {
            queuedStart = exchange.getProperty(PROPERTY_EXCHANGE_QUEUED_TIMESTAMP, 0L, Long.class);
            exchange.removeProperty(PROPERTY_EXCHANGE_QUEUED_TIMESTAMP);
        }
        State state = exchange.getProperty(PROPERTY_EXCHANGE_STATE, (Object)State.SYNC, State.class);
        exchange.removeProperty(PROPERTY_EXCHANGE_STATE);
        boolean doneSync = state == State.SYNC || state == State.ASYNC_REJECTED;
        try {
            if (!this.isRunAllowed()) {
                throw new RejectedExecutionException("Run is not allowed");
            }
            this.calculateAndSetMaxRequestsPerPeriod(exchange);
            ThrottlePermit permit = (ThrottlePermit)this.delayQueue.poll();
            if (permit == null) {
                if (this.isRejectExecution()) {
                    throw new ThrottlerRejectedExecutionException("Exceeded the max throttle rate of " + this.throttleRate + " within " + this.timePeriodMillis + "ms");
                }
                if (this.isAsyncDelayed() && !exchange.isTransacted() && state == State.SYNC) {
                    this.log.debug("Throttle rate exceeded but AsyncDelayed enabled, so queueing for async processing, exchangeId: {}", (Object)exchange.getExchangeId());
                    return this.processAsynchronously(exchange, callback);
                }
                long start = 0L;
                long elapsed = 0L;
                if (this.log.isTraceEnabled()) {
                    start = System.currentTimeMillis();
                }
                permit = (ThrottlePermit)this.delayQueue.take();
                if (this.log.isTraceEnabled()) {
                    elapsed = System.currentTimeMillis() - start;
                }
                this.enqueuePermit(permit, exchange);
                if (state == State.ASYNC) {
                    if (this.log.isTraceEnabled()) {
                        long queuedTime = start - queuedStart;
                        this.log.trace("Queued for {}ms, Throttled for {}ms, exchangeId: {}", new Object[]{queuedTime, elapsed, exchange.getExchangeId()});
                    }
                } else {
                    this.log.trace("Throttled for {}ms, exchangeId: {}", (Object)elapsed, (Object)exchange.getExchangeId());
                }
            } else {
                this.enqueuePermit(permit, exchange);
                if (state == State.ASYNC) {
                    if (this.log.isTraceEnabled()) {
                        long queuedTime = System.currentTimeMillis() - queuedStart;
                        this.log.trace("Queued for {}ms, No throttling applied (throttle cleared while queued), for exchangeId: {}", (Object)queuedTime, (Object)exchange.getExchangeId());
                    }
                } else {
                    this.log.trace("No throttling applied to exchangeId: {}", (Object)exchange.getExchangeId());
                }
            }
            if (this.processor != null) {
                if (doneSync) {
                    return this.processor.process(exchange, callback);
                }
                AsyncProcessorHelper.process(this.processor, exchange);
            }
            callback.done(doneSync);
            return doneSync;
        }
        catch (InterruptedException e) {
            boolean forceShutdown = exchange.getContext().getShutdownStrategy().forceShutdown(this);
            if (forceShutdown) {
                String msg = "Run not allowed as ShutdownStrategy is forcing shutting down, will reject executing exchange: " + exchange;
                this.log.debug(msg);
                exchange.setException(new RejectedExecutionException(msg, e));
            } else {
                exchange.setException(e);
            }
            callback.done(doneSync);
            return doneSync;
        }
        catch (Throwable t) {
            exchange.setException(t);
            callback.done(doneSync);
            return doneSync;
        }
    }

    protected boolean processAsynchronously(final Exchange exchange, final AsyncCallback callback) {
        try {
            if (this.log.isTraceEnabled()) {
                exchange.setProperty(PROPERTY_EXCHANGE_QUEUED_TIMESTAMP, System.currentTimeMillis());
            }
            exchange.setProperty(PROPERTY_EXCHANGE_STATE, (Object)State.ASYNC);
            this.asyncExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    Throttler.this.process(exchange, callback);
                }
            });
            return false;
        }
        catch (RejectedExecutionException e) {
            if (this.isCallerRunsWhenRejected()) {
                this.log.debug("AsyncExecutor is full, rejected exchange will run in the current thread, exchangeId: {}", (Object)exchange.getExchangeId());
                exchange.setProperty(PROPERTY_EXCHANGE_STATE, (Object)State.ASYNC_REJECTED);
                return this.process(exchange, callback);
            }
            throw e;
        }
    }

    protected void enqueuePermit(ThrottlePermit permit, Exchange exchange) {
        permit.setDelayMs(this.getTimePeriodMillis());
        this.delayQueue.put(permit);
        if (this.log.isTraceEnabled()) {
            this.log.trace("Permit released, for exchangeId: {}", (Object)exchange.getExchangeId());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void calculateAndSetMaxRequestsPerPeriod(Exchange exchange) throws Exception {
        Integer newThrottle = this.maxRequestsPerPeriodExpression.evaluate(exchange, Integer.class);
        if (newThrottle != null && newThrottle < 0) {
            throw new IllegalStateException("The maximumRequestsPerPeriod must be a positive number, was: " + newThrottle);
        }
        Throttler throttler = this;
        synchronized (throttler) {
            if (newThrottle == null && this.throttleRate == 0) {
                throw new RuntimeExchangeException("The maxRequestsPerPeriodExpression was evaluated as null: " + this.maxRequestsPerPeriodExpression, exchange);
            }
            if (newThrottle != null && newThrottle != this.throttleRate) {
                if (this.throttleRate > newThrottle) {
                    for (int delta = this.throttleRate - newThrottle; delta > 0; --delta) {
                        this.delayQueue.take();
                        this.log.trace("Permit discarded due to throttling rate decrease, triggered by ExchangeId: {}", (Object)exchange.getExchangeId());
                    }
                    this.log.debug("Throttle rate decreased from {} to {}, triggered by ExchangeId: {}", new Object[]{this.throttleRate, newThrottle, exchange.getExchangeId()});
                } else if (newThrottle > this.throttleRate) {
                    int delta = newThrottle - this.throttleRate;
                    for (int i = 0; i < delta; ++i) {
                        this.delayQueue.put(new ThrottlePermit(-1L));
                    }
                    if (this.throttleRate == 0) {
                        this.log.debug("Initial throttle rate set to {}, triggered by ExchangeId: {}", (Object)newThrottle, (Object)exchange.getExchangeId());
                    } else {
                        this.log.debug("Throttle rate increase from {} to {}, triggered by ExchangeId: {}", new Object[]{this.throttleRate, newThrottle, exchange.getExchangeId()});
                    }
                }
                this.throttleRate = newThrottle;
            }
        }
    }

    @Override
    protected void doStart() throws Exception {
        if (this.isAsyncDelayed()) {
            ObjectHelper.notNull(this.asyncExecutor, "executorService", this);
        }
        super.doStart();
    }

    @Override
    protected void doShutdown() throws Exception {
        if (this.shutdownAsyncExecutor && this.asyncExecutor != null) {
            this.camelContext.getExecutorServiceManager().shutdownNow(this.asyncExecutor);
        }
        super.doShutdown();
    }

    public boolean isRejectExecution() {
        return this.rejectExecution;
    }

    public void setRejectExecution(boolean rejectExecution) {
        this.rejectExecution = rejectExecution;
    }

    public boolean isAsyncDelayed() {
        return this.asyncDelayed;
    }

    public void setAsyncDelayed(boolean asyncDelayed) {
        this.asyncDelayed = asyncDelayed;
    }

    public boolean isCallerRunsWhenRejected() {
        return this.callerRunsWhenRejected;
    }

    public void setCallerRunsWhenRejected(boolean callerRunsWhenRejected) {
        this.callerRunsWhenRejected = callerRunsWhenRejected;
    }

    @Override
    public String getId() {
        return this.id;
    }

    @Override
    public void setId(String id) {
        this.id = id;
    }

    public void setMaximumRequestsPerPeriodExpression(Expression maxRequestsPerPeriodExpression) {
        this.maxRequestsPerPeriodExpression = maxRequestsPerPeriodExpression;
    }

    public Expression getMaximumRequestsPerPeriodExpression() {
        return this.maxRequestsPerPeriodExpression;
    }

    public int getCurrentMaximumRequestsPerPeriod() {
        return this.throttleRate;
    }

    public void setTimePeriodMillis(long timePeriodMillis) {
        this.timePeriodMillis = timePeriodMillis;
    }

    public long getTimePeriodMillis() {
        return this.timePeriodMillis;
    }

    @Override
    public String getTraceLabel() {
        return "throttle[" + this.maxRequestsPerPeriodExpression + " per: " + this.timePeriodMillis + "]";
    }

    @Override
    public String toString() {
        return "Throttler[requests: " + this.maxRequestsPerPeriodExpression + " per: " + this.timePeriodMillis + " (ms) to: " + this.getProcessor() + "]";
    }

    private class ThrottlePermit
    implements Delayed {
        private volatile long scheduledTime;

        ThrottlePermit(long delayMs) {
            this.setDelayMs(delayMs);
        }

        public void setDelayMs(long delayMs) {
            this.scheduledTime = System.currentTimeMillis() + delayMs;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.scheduledTime - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            return (int)(this.getDelay(TimeUnit.MILLISECONDS) - o.getDelay(TimeUnit.MILLISECONDS));
        }
    }

    private static enum State {
        SYNC,
        ASYNC,
        ASYNC_REJECTED;

    }
}

