import cloneDeep from "lodash/cloneDeep";
import get from "lodash/get";
import merge from "lodash/merge";
import unset from "lodash/unset";
import set from "lodash/set";
export function parseTaskSchema(formSchema) {
    const schema = cloneDeep(formSchema);
    const assignments = {
        inputs: [],
        outputs: [],
    };
    if (!formSchema.properties) {
        return {
            schema,
            assignments,
        };
    }
    for (const key of Object.keys(schema.properties)) {
        const property = schema.properties[key];
        const assignmetChecker = property["allOf"] ? checkAssignmentForAllOf : checkAssignment;
        if (assignmetChecker(property, "input")) {
            assignments.inputs.push(key);
        }
        if (assignmetChecker(property, "output")) {
            assignments.outputs.push(key);
        }
        if (!assignments.outputs.includes(key)) {
            set(property, "uniforms.disabled", true);
        }
    }
    return {
        schema,
        assignments,
    };
}
function checkAssignmentForAllOf(property, assignmentExpr) {
    const allOf = property["allOf"];
    const assignment = allOf.find((value) => value[assignmentExpr]);
    if (assignment) {
        const index = allOf.indexOf(assignment);
        allOf.splice(index, 1);
        return true;
    }
    return false;
}
function checkAssignment(property, assignmentExpr) {
    if (property[assignmentExpr]) {
        delete property[assignmentExpr];
        return true;
    }
    return false;
}
export function readSchemaAssignments(formSchema, isUniforms = true) {
    const assignments = {
        inputs: [],
        outputs: [],
    };
    if (!formSchema.properties) {
        return assignments;
    }
    for (const key of Object.keys(formSchema.properties)) {
        const property = formSchema.properties[key];
        if (get(property, "input", false)) {
            assignments.inputs.push(key);
            if (isUniforms) {
                unset(property, "input");
            }
        }
        if (get(property, "output", false)) {
            assignments.outputs.push(key);
            if (isUniforms) {
                unset(property, "output");
            }
        }
        if (isUniforms && !assignments.outputs.includes(key)) {
            const uniforms = get(property, "uniforms", {});
            uniforms.disabled = true;
            set(property, "uniforms", uniforms);
        }
    }
    return assignments;
}
function toJSON(value) {
    if (value) {
        try {
            return JSON.parse(value);
        }
        catch (e) {
        }
    }
    return {};
}
export function generateFormData(userTask) {
    const taskInputs = toJSON(userTask.inputs);
    if (!userTask.outputs) {
        return taskInputs;
    }
    const taskOutputs = toJSON(userTask.outputs);
    return merge(taskInputs, taskOutputs);
}
//# sourceMappingURL=TaskFormDataUtils.js.map