import { Fragment as _Fragment, jsx as _jsx } from "react/jsx-runtime";
import * as RF from "reactflow";
import { useEffect } from "react";
import { DMN_EDITOR_DIAGRAM_CLIPBOARD_MIME_TYPE, buildClipboardFromDiagram, getClipboard, } from "../clipboard/Clipboard";
import { buildXmlHref, parseXmlHref } from "@kie-tools/dmn-marshaller/dist/xml/xmlHrefs";
import { getNewDmnIdRandomizer } from "@kie-tools/dmn-marshaller/dist/idRandomizer/dmnIdRandomizer";
import { NodeNature, nodeNatures } from "../mutations/NodeNature";
import { addOrGetDrd } from "../mutations/addOrGetDrd";
import { addStandaloneNode } from "../mutations/addStandaloneNode";
import { EdgeDeletionMode, deleteEdge } from "../mutations/deleteEdge";
import { NodeDeletionMode, canRemoveNodeFromDrdOnly, deleteNode } from "../mutations/deleteNode";
import { repopulateInputDataAndDecisionsOnAllDecisionServices } from "../mutations/repopulateInputDataAndDecisionsOnDecisionService";
import { useDmnEditorStoreApi } from "../store/StoreContext";
import { CONTAINER_NODES_DESIRABLE_PADDING, getBounds } from "./maths/DmnMaths";
import { NODE_TYPES } from "./nodes/NodeTypes";
import { useExternalModels } from "../includedModels/DmnEditorDependenciesContext";
import { DEFAULT_VIEWPORT } from "./Diagram";
import { useCommands } from "../commands/CommandsContextProvider";
export function DiagramCommands(props) {
    const rfStoreApi = RF.useStoreApi();
    const dmnEditorStoreApi = useDmnEditorStoreApi();
    const { commandsRef } = useCommands();
    const { externalModelsByNamespace } = useExternalModels();
    const rf = RF.useReactFlow();
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.cancelAction = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Canceling action...");
            rfStoreApi.setState((rfState) => {
                var _a, _b;
                if (rfState.connectionNodeId) {
                    rfState.cancelConnection();
                    dmnEditorStoreApi.setState((state) => {
                        state.diagram.ongoingConnection = undefined;
                    });
                }
                else {
                    (_b = (_a = document.activeElement) === null || _a === void 0 ? void 0 : _a.blur) === null || _b === void 0 ? void 0 : _b.call(_a);
                }
                return rfState;
            });
        };
    }, [dmnEditorStoreApi, commandsRef, rfStoreApi]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.resetPosition = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Reseting position...");
            rf.setViewport(DEFAULT_VIEWPORT, { duration: 200 });
        };
    }, [commandsRef, rf]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.focusOnSelection = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Focusing on selected bounds...");
            const selectedNodes = rf.getNodes().filter((s) => s.selected);
            if (selectedNodes.length <= 0) {
                return;
            }
            const bounds = getBounds({
                nodes: selectedNodes,
                padding: 100,
            });
            rf.fitBounds({
                x: bounds["@_x"],
                y: bounds["@_y"],
                width: bounds["@_width"],
                height: bounds["@_height"],
            }, { duration: 200 });
        };
    }, [commandsRef, rf]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.cut = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Cutting selected nodes...");
            const { clipboard, copiedEdgesById, danglingEdgesById, copiedNodesById } = buildClipboardFromDiagram(rfStoreApi.getState(), dmnEditorStoreApi.getState());
            navigator.clipboard.writeText(JSON.stringify(clipboard)).then(() => {
                dmnEditorStoreApi.setState((state) => {
                    [...copiedEdgesById.values(), ...danglingEdgesById.values()].forEach((edge) => {
                        deleteEdge({
                            definitions: state.dmn.model.definitions,
                            drdIndex: state.computed(state).getDrdIndex(),
                            edge: { id: edge.id, dmnObject: edge.data.dmnObject },
                            mode: EdgeDeletionMode.FROM_DRG_AND_ALL_DRDS,
                            externalModelsByNamespace,
                        });
                        state.dispatch(state).diagram.setEdgeStatus(edge.id, {
                            selected: false,
                            draggingWaypoint: false,
                        });
                    });
                    rfStoreApi
                        .getState()
                        .getNodes()
                        .forEach((node) => {
                        var _a, _b;
                        if (copiedNodesById.has(node.id)) {
                            deleteNode({
                                __readonly_drgEdges: state.computed(state).getDiagramData(externalModelsByNamespace).drgEdges,
                                definitions: state.dmn.model.definitions,
                                __readonly_drdIndex: state.computed(state).getDrdIndex(),
                                __readonly_dmnObjectNamespace: (_a = node.data.dmnObjectNamespace) !== null && _a !== void 0 ? _a : state.dmn.model.definitions["@_namespace"],
                                __readonly_dmnObjectQName: node.data.dmnObjectQName,
                                __readonly_dmnObjectId: (_b = node.data.dmnObject) === null || _b === void 0 ? void 0 : _b["@_id"],
                                __readonly_nodeNature: nodeNatures[node.type],
                                __readonly_mode: NodeDeletionMode.FROM_DRG_AND_ALL_DRDS,
                                __readonly_externalDmnsIndex: state
                                    .computed(state)
                                    .getDirectlyIncludedExternalModelsByNamespace(externalModelsByNamespace).dmns,
                                __readonly_externalModelsByNamespace: externalModelsByNamespace,
                            });
                            state.dispatch(state).diagram.setNodeStatus(node.id, {
                                selected: false,
                                dragging: false,
                                resizing: false,
                            });
                        }
                    });
                });
            });
        };
    }, [dmnEditorStoreApi, externalModelsByNamespace, commandsRef, rfStoreApi]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.copy = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Copying selected nodes...");
            const { clipboard } = buildClipboardFromDiagram(rfStoreApi.getState(), dmnEditorStoreApi.getState());
            navigator.clipboard.writeText(JSON.stringify(clipboard));
        };
    }, [dmnEditorStoreApi, commandsRef, rfStoreApi]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.paste = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Pasting nodes...");
            navigator.clipboard.readText().then((text) => {
                const clipboard = getClipboard(text, DMN_EDITOR_DIAGRAM_CLIPBOARD_MIME_TYPE);
                if (!clipboard) {
                    return;
                }
                getNewDmnIdRandomizer()
                    .ack({
                    json: clipboard.drgElements,
                    type: "DMN15__tDefinitions",
                    attr: "drgElement",
                })
                    .ack({
                    json: clipboard.artifacts,
                    type: "DMN15__tDefinitions",
                    attr: "artifact",
                })
                    .ack({
                    json: clipboard.shapes,
                    type: "DMNDI15__DMNDiagram",
                    attr: "dmndi:DMNDiagramElement",
                    __$$element: "dmndi:DMNShape",
                })
                    .ack({
                    json: clipboard.edges,
                    type: "DMNDI15__DMNDiagram",
                    attr: "dmndi:DMNDiagramElement",
                    __$$element: "dmndi:DMNEdge",
                })
                    .ack({
                    json: clipboard.widths,
                    type: "KIE__tComponentsWidthsExtension",
                    attr: "kie:ComponentWidths",
                })
                    .randomize({ skipAlreadyAttributedIds: false });
                dmnEditorStoreApi.setState((state) => {
                    var _a, _b;
                    var _c, _d;
                    (_a = (_c = state.dmn.model.definitions).drgElement) !== null && _a !== void 0 ? _a : (_c.drgElement = []);
                    state.dmn.model.definitions.drgElement.push(...clipboard.drgElements);
                    (_b = (_d = state.dmn.model.definitions).artifact) !== null && _b !== void 0 ? _b : (_d.artifact = []);
                    state.dmn.model.definitions.artifact.push(...clipboard.artifacts);
                    const { diagramElements, widths } = addOrGetDrd({
                        definitions: state.dmn.model.definitions,
                        drdIndex: state.computed(state).getDrdIndex(),
                    });
                    diagramElements.push(...clipboard.shapes.map((s) => ({
                        ...s,
                        __$$element: "dmndi:DMNShape",
                    })));
                    diagramElements.push(...clipboard.edges.map((s) => ({
                        ...s,
                        __$$element: "dmndi:DMNEdge",
                    })));
                    widths.push(...clipboard.widths);
                    repopulateInputDataAndDecisionsOnAllDecisionServices({
                        definitions: state.dmn.model.definitions,
                        externalModelsByNamespace,
                    });
                    state.diagram._selectedNodes = [...clipboard.drgElements, ...clipboard.artifacts].map((s) => buildXmlHref({ id: s["@_id"] }));
                    if (state.diagram._selectedNodes.length === 1) {
                        state.focus.consumableId = parseXmlHref(state.diagram._selectedNodes[0]).id;
                    }
                });
            });
        };
    }, [dmnEditorStoreApi, commandsRef, externalModelsByNamespace]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.selectAll = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Selecting/Deselecting nodes...");
            const allNodeIds = rfStoreApi
                .getState()
                .getNodes()
                .map((s) => s.id);
            const allEdgeIds = rfStoreApi.getState().edges.map((s) => s.id);
            dmnEditorStoreApi.setState((state) => {
                const allSelectedNodesSet = new Set(state.diagram._selectedNodes);
                const allSelectedEdgesSet = new Set(state.diagram._selectedEdges);
                if (allNodeIds.every((id) => allSelectedNodesSet.has(id) && allEdgeIds.every((id) => allSelectedEdgesSet.has(id)))) {
                    state.diagram._selectedNodes = [];
                    state.diagram._selectedEdges = [];
                }
                else {
                    state.diagram._selectedNodes = allNodeIds;
                    state.diagram._selectedEdges = allEdgeIds;
                }
            });
        };
    }, [dmnEditorStoreApi, commandsRef, rfStoreApi]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.createGroup = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Grouping nodes...");
            const selectedNodes = rf.getNodes().filter((s) => s.selected);
            if (selectedNodes.length <= 0) {
                return;
            }
            dmnEditorStoreApi.setState((state) => {
                if (state.diagram._selectedNodes.length <= 0) {
                    return;
                }
                const { href: newNodeId } = addStandaloneNode({
                    definitions: state.dmn.model.definitions,
                    drdIndex: state.computed(state).getDrdIndex(),
                    newNode: {
                        type: NODE_TYPES.group,
                        bounds: getBounds({
                            nodes: selectedNodes,
                            padding: CONTAINER_NODES_DESIRABLE_PADDING,
                        }),
                    },
                    externalModelsByNamespace,
                });
                state.dispatch(state).diagram.setNodeStatus(newNodeId, { selected: true });
            });
        };
    }, [dmnEditorStoreApi, commandsRef, rf, externalModelsByNamespace]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.toggleHierarchyHighlight = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Toggle hierarchy highlights...");
            dmnEditorStoreApi.setState((state) => {
                state.diagram.overlays.enableNodeHierarchyHighlight = !state.diagram.overlays.enableNodeHierarchyHighlight;
            });
        };
    }, [dmnEditorStoreApi, commandsRef]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.togglePropertiesPanel = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Toggle properties panel...");
            dmnEditorStoreApi.setState((state) => {
                state.diagram.propertiesPanel.isOpen = !state.diagram.propertiesPanel.isOpen;
            });
        };
    }, [dmnEditorStoreApi, commandsRef]);
    useEffect(() => {
        if (!commandsRef.current) {
            return;
        }
        commandsRef.current.hideFromDrd = async () => {
            console.debug("DMN DIAGRAM: COMMANDS: Hide node from DRD...");
            const nodesById = rf
                .getNodes()
                .reduce((acc, s) => acc.set(s.id, s), new Map());
            dmnEditorStoreApi.setState((state) => {
                var _a, _b, _c, _d, _e, _f;
                const selectedNodeIds = new Set(state.diagram._selectedNodes);
                for (const edge of rf.getEdges()) {
                    if ((selectedNodeIds.has(edge.source) &&
                        canRemoveNodeFromDrdOnly({
                            __readonly_externalDmnsIndex: state
                                .computed(state)
                                .getDirectlyIncludedExternalModelsByNamespace(externalModelsByNamespace).dmns,
                            definitions: state.dmn.model.definitions,
                            __readonly_drdIndex: state.computed(state).getDrdIndex(),
                            __readonly_dmnObjectNamespace: (_a = nodesById.get(edge.source).data.dmnObjectNamespace) !== null && _a !== void 0 ? _a : state.dmn.model.definitions["@_namespace"],
                            __readonly_dmnObjectId: (_b = nodesById.get(edge.source).data.dmnObject) === null || _b === void 0 ? void 0 : _b["@_id"],
                        })) ||
                        (selectedNodeIds.has(edge.target) &&
                            canRemoveNodeFromDrdOnly({
                                __readonly_externalDmnsIndex: state
                                    .computed(state)
                                    .getDirectlyIncludedExternalModelsByNamespace(externalModelsByNamespace).dmns,
                                definitions: state.dmn.model.definitions,
                                __readonly_drdIndex: state.computed(state).getDrdIndex(),
                                __readonly_dmnObjectNamespace: (_c = nodesById.get(edge.target).data.dmnObjectNamespace) !== null && _c !== void 0 ? _c : state.dmn.model.definitions["@_namespace"],
                                __readonly_dmnObjectId: (_d = nodesById.get(edge.target).data.dmnObject) === null || _d === void 0 ? void 0 : _d["@_id"],
                            }))) {
                        deleteEdge({
                            definitions: state.dmn.model.definitions,
                            drdIndex: state.computed(state).getDrdIndex(),
                            edge: { id: edge.id, dmnObject: edge.data.dmnObject },
                            mode: EdgeDeletionMode.FROM_CURRENT_DRD_ONLY,
                            externalModelsByNamespace,
                        });
                        state.dispatch(state).diagram.setEdgeStatus(edge.id, {
                            selected: false,
                            draggingWaypoint: false,
                        });
                    }
                }
                for (const node of rf.getNodes().filter((s) => s.selected)) {
                    if (nodeNatures[node.type] === NodeNature.ARTIFACT) {
                        continue;
                    }
                    const { deletedDmnShapeOnCurrentDrd: deletedShape } = deleteNode({
                        definitions: state.dmn.model.definitions,
                        __readonly_drgEdges: [],
                        __readonly_externalDmnsIndex: state
                            .computed(state)
                            .getDirectlyIncludedExternalModelsByNamespace(externalModelsByNamespace).dmns,
                        __readonly_drdIndex: state.computed(state).getDrdIndex(),
                        __readonly_dmnObjectNamespace: (_e = node.data.dmnObjectNamespace) !== null && _e !== void 0 ? _e : state.dmn.model.definitions["@_namespace"],
                        __readonly_dmnObjectQName: node.data.dmnObjectQName,
                        __readonly_dmnObjectId: (_f = node.data.dmnObject) === null || _f === void 0 ? void 0 : _f["@_id"],
                        __readonly_nodeNature: nodeNatures[node.type],
                        __readonly_mode: NodeDeletionMode.FROM_CURRENT_DRD_ONLY,
                        __readonly_externalModelsByNamespace: externalModelsByNamespace,
                    });
                    if (deletedShape) {
                        state.dispatch(state).diagram.setNodeStatus(node.id, {
                            selected: false,
                            dragging: false,
                            resizing: false,
                        });
                    }
                }
            });
        };
    }, [dmnEditorStoreApi, externalModelsByNamespace, commandsRef, rf]);
    return _jsx(_Fragment, {});
}
//# sourceMappingURL=DiagramCommands.js.map