import { CompoundPredicate, False, SimplePredicate, SimpleSetPredicate, True, } from "@kie-tools/pmml-editor-marshaller";
const SimplePredicateOperatorMap = new Map([
    ["equal", "="],
    ["notEqual", "<>"],
    ["lessThan", "<"],
    ["lessOrEqual", "<="],
    ["greaterThan", ">"],
    ["greaterOrEqual", ">="],
    ["isMissing", "isMissing"],
    ["isNotMissing", "isNotMissing"],
]);
export const toText = (predicate, fields) => {
    const fieldToDataType = new Map(fields.map((field) => [field.name, field.dataType]));
    return _toText(predicate, fieldToDataType, 0);
};
const _toText = (predicate, fieldToDataType, nesting) => {
    var _a;
    if (predicate instanceof True) {
        return "True";
    }
    else if (predicate instanceof False) {
        return "False";
    }
    else if (predicate instanceof SimpleSetPredicate) {
        const ssp = predicate;
        return `${ssp.field.toString()} ${ssp.booleanOperator} ${ssp.Array.toString()} `;
    }
    else if (predicate instanceof SimplePredicate) {
        const sp = predicate;
        return `${sp.field.toString()} ${SimplePredicateOperatorMap.get(sp.operator)} ${_value(sp.field, sp.value, fieldToDataType)}`;
    }
    else if (predicate instanceof CompoundPredicate) {
        const cp = predicate;
        let text = "";
        const children = [];
        (_a = cp.predicates) === null || _a === void 0 ? void 0 : _a.forEach((p) => children.push(_toText(p, fieldToDataType, nesting + 1)));
        text = text + children.join(" " + cp.booleanOperator + " ");
        return text;
    }
    return "";
};
const TruePredicate = () => {
    const predicate = new True({});
    predicate._type = "True";
    return predicate;
};
const FalsePredicate = () => {
    const predicate = new False({});
    predicate._type = "False";
    return predicate;
};
const UnarySimplePredicate = (field, operator) => {
    const predicate = new SimplePredicate({ field: field, operator: operator });
    predicate._type = "SimplePredicate";
    return predicate;
};
const BinarySimplePredicate = (field, operator, value) => {
    const predicate = new SimplePredicate({ field: field, operator: operator, value: value });
    predicate._type = "SimplePredicate";
    return predicate;
};
const SimpleCompoundPredicate = (predicate1, predicate2, booleanOperator) => {
    const predicate = new CompoundPredicate({ predicates: [predicate1, predicate2], booleanOperator });
    predicate._type = "CompoundPredicate";
    return predicate;
};
const _value = (field, value, fieldToDataType) => {
    if (value === undefined) {
        return "";
    }
    const dataType = fieldToDataType.get(field);
    if (dataType === "string") {
        return `"${value}"`;
    }
    return value.toString();
};
const _operator = (lookup) => {
    const lookups = SimplePredicateOperatorMap.entries();
    for (const [pmmlOperator, operator] of lookups) {
        if (operator === lookup) {
            return pmmlOperator;
        }
    }
};
export const fromText = (text) => {
    if (text === undefined) {
        return undefined;
    }
    text = text.trim();
    if (text === "") {
        return undefined;
    }
    const regTrue = /^True$/gim;
    const regFalse = /^False$/gim;
    const regUnaryOperator = /^(\S+)\s+(isMissing|isNotMissing)\s*$/gm;
    const regBinaryOperator = /^(\S+)\s*(=|>|<|<=|>=|<>)\s*"?(\S+)"?$/gm;
    const regSimpleCompound = /^(.*)\s*(\band\b|\bxor\b|\bor\b)\s*(.*)$/gm;
    if (regTrue.test(text)) {
        return TruePredicate();
    }
    if (regFalse.test(text)) {
        return FalsePredicate();
    }
    const unaryMatches = regUnaryOperator.exec(text);
    if (unaryMatches !== null) {
        return UnarySimplePredicate(unaryMatches[1], unaryMatches[2]);
    }
    const binaryMatches = regBinaryOperator.exec(text);
    if (binaryMatches !== null) {
        return BinarySimplePredicate(binaryMatches[1], _operator(binaryMatches[2]), binaryMatches[3]);
    }
    const compoundMatches = regSimpleCompound.exec(text);
    if (compoundMatches !== null) {
        const predicate1 = fromText(compoundMatches[1]);
        const predicate2 = fromText(compoundMatches[3]);
        const booleanOperator = compoundMatches[2];
        if (predicate1 !== undefined && predicate2 !== undefined && booleanOperator !== undefined) {
            return SimpleCompoundPredicate(predicate1, predicate2, booleanOperator);
        }
    }
    return TruePredicate();
};
//# sourceMappingURL=PredicateConverter.js.map