"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HistoryService = void 0;
var immer_1 = require("immer");
var lodash_1 = require("lodash");
var HistoryService = (function () {
    function HistoryService(listeners) {
        var _this = this;
        this.listeners = listeners;
        this.pending = new Array();
        this.history = {
            changes: [],
            index: 0,
        };
        this.batch = function (state, path, recipe, validate) {
            _this.pending.push({ state: state, path: path, recipe: recipe, validate: validate });
        };
        this.commit = function (state) {
            if (state === undefined) {
                return;
            }
            var newState = _this.mutate(state, null, function (draft) {
                _this.pending.forEach(function (be) {
                    var segment = be.path === null ? draft : (0, lodash_1.get)(draft, be.path.path);
                    be.recipe(segment);
                });
            });
            _this.pending.forEach(function (be) {
                if (be.validate !== undefined) {
                    be.validate(newState);
                }
            });
            _this.listeners.forEach(function (listener) { return listener("Command".concat(_this.history.index)); });
            _this.pending = new Array();
            return newState;
        };
        this.mutate = function (state, path, recipe) {
            if (_this.history.index < _this.history.changes.length) {
                _this.history.changes = _this.history.changes.slice(0, _this.history.index);
            }
            var newState = (0, immer_1.produce)(state, recipe, function (patches, inversePatches) {
                _this.history.changes.push({ path: path, change: patches, reverse: inversePatches });
                _this.history.index = _this.history.changes.length;
            });
            return newState;
        };
        this.undo = function (state) {
            if (_this.history.index > 0) {
                var change = _this.history.changes[--_this.history.index];
                return _this.apply(state, change.path, change.reverse);
            }
            return state;
        };
        this.redo = function (state) {
            if (_this.history.index < _this.history.changes.length) {
                var change = _this.history.changes[_this.history.index++];
                return _this.apply(state, change.path, change.change);
            }
            return state;
        };
        this.index = function () {
            return _this.history.index;
        };
        this.changes = function () {
            return _this.history.changes;
        };
        this.apply = function (state, path, patch) {
            if (path === null) {
                return (0, immer_1.applyPatches)(state, patch);
            }
            var branch = (0, lodash_1.get)(state, path.path);
            var branchUndone = (0, immer_1.applyPatches)(branch, patch);
            var newState = (0, lodash_1.cloneDeep)(state);
            return (0, lodash_1.set)(newState, path.path, branchUndone);
        };
    }
    return HistoryService;
}());
exports.HistoryService = HistoryService;
//# sourceMappingURL=HistoryProvider.js.map