/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.extensions.gcp.storage;

import com.google.api.client.util.DateTime;
import com.google.api.services.storage.model.Objects;
import com.google.api.services.storage.model.StorageObject;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.beam.sdk.extensions.gcp.options.GcsOptions;
import org.apache.beam.sdk.extensions.gcp.storage.GcsCreateOptions;
import org.apache.beam.sdk.extensions.gcp.storage.GcsResourceId;
import org.apache.beam.sdk.extensions.gcp.util.GcsUtil;
import org.apache.beam.sdk.extensions.gcp.util.gcsfs.GcsPath;
import org.apache.beam.sdk.io.FileSystem;
import org.apache.beam.sdk.io.FileSystemUtils;
import org.apache.beam.sdk.io.fs.CreateOptions;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.io.fs.MoveOptions;
import org.apache.beam.sdk.io.fs.ResourceId;
import org.apache.beam.sdk.metrics.Counter;
import org.apache.beam.sdk.metrics.Metrics;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.annotations.VisibleForTesting;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.MoreObjects;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Preconditions;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Stopwatch;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.FluentIterable;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.ImmutableList;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.Lists;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GcsFileSystem
extends FileSystem<GcsResourceId> {
    private static final Logger LOG = LoggerFactory.getLogger(GcsFileSystem.class);
    private final GcsOptions options;
    private Counter numCopies;
    private Counter numRenames;
    private Counter copyTimeMsec;
    private Counter renameTimeMsec;

    GcsFileSystem(GcsOptions options) {
        this.options = (GcsOptions)Preconditions.checkNotNull((Object)options, (Object)"options");
        if (options.getGcsPerformanceMetrics().booleanValue()) {
            this.numCopies = Metrics.counter(GcsFileSystem.class, (String)"num_copies");
            this.copyTimeMsec = Metrics.counter(GcsFileSystem.class, (String)"copy_time_msec");
            this.numRenames = Metrics.counter(GcsFileSystem.class, (String)"num_renames");
            this.renameTimeMsec = Metrics.counter(GcsFileSystem.class, (String)"rename_time_msec");
        }
    }

    protected List<MatchResult> match(List<String> specs) throws IOException {
        List<GcsPath> gcsPaths = this.toGcsPaths(specs);
        ArrayList globs = Lists.newArrayList();
        ArrayList nonGlobs = Lists.newArrayList();
        ArrayList isGlobBooleans = Lists.newArrayList();
        for (GcsPath path : gcsPaths) {
            if (GcsUtil.isWildcard(path)) {
                globs.add(path);
                isGlobBooleans.add(true);
                continue;
            }
            nonGlobs.add(path);
            isGlobBooleans.add(false);
        }
        Iterator<MatchResult> globsMatchResults = this.matchGlobs(globs).iterator();
        Iterator<MatchResult> nonGlobsMatchResults = this.matchNonGlobs(nonGlobs).iterator();
        ImmutableList.Builder ret = ImmutableList.builder();
        for (Boolean isGlob : isGlobBooleans) {
            if (isGlob.booleanValue()) {
                Preconditions.checkState((boolean)globsMatchResults.hasNext(), (String)"Expect globsMatchResults has next: %s", (Object)globs);
                ret.add((Object)globsMatchResults.next());
                continue;
            }
            Preconditions.checkState((boolean)nonGlobsMatchResults.hasNext(), (String)"Expect nonGlobsMatchResults has next: %s", (Object)nonGlobs);
            ret.add((Object)nonGlobsMatchResults.next());
        }
        Preconditions.checkState((!globsMatchResults.hasNext() ? 1 : 0) != 0, (Object)"Internal error encountered in GcsFilesystem: expected no more elements in globsMatchResults.");
        Preconditions.checkState((!nonGlobsMatchResults.hasNext() ? 1 : 0) != 0, (Object)"Internal error encountered in GcsFilesystem: expected no more elements in globsMatchResults.");
        return ret.build();
    }

    protected WritableByteChannel create(GcsResourceId resourceId, CreateOptions createOptions) throws IOException {
        GcsUtil.CreateOptions.Builder builder = GcsUtil.CreateOptions.builder().setContentType(createOptions.mimeType()).setExpectFileToNotExist(createOptions.expectFileToNotExist());
        if (createOptions instanceof GcsCreateOptions) {
            builder = builder.setUploadBufferSizeBytes(((GcsCreateOptions)createOptions).gcsUploadBufferSizeBytes());
        }
        return this.options.getGcsUtil().create(resourceId.getGcsPath(), builder.build());
    }

    protected ReadableByteChannel open(GcsResourceId resourceId) throws IOException {
        return this.options.getGcsUtil().open(resourceId.getGcsPath());
    }

    protected void rename(List<GcsResourceId> srcResourceIds, List<GcsResourceId> destResourceIds, MoveOptions ... moveOptions) throws IOException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        this.options.getGcsUtil().rename(this.toFilenames(srcResourceIds), this.toFilenames(destResourceIds), moveOptions);
        stopwatch.stop();
        if (this.options.getGcsPerformanceMetrics().booleanValue()) {
            this.numRenames.inc((long)srcResourceIds.size());
            this.renameTimeMsec.inc(stopwatch.elapsed(TimeUnit.MILLISECONDS));
        }
    }

    protected void delete(Collection<GcsResourceId> resourceIds) throws IOException {
        this.options.getGcsUtil().remove(this.toFilenames(resourceIds));
    }

    protected GcsResourceId matchNewResource(String singleResourceSpec, boolean isDirectory) {
        if (isDirectory) {
            if (!singleResourceSpec.endsWith("/")) {
                singleResourceSpec = singleResourceSpec + "/";
            }
        } else {
            Preconditions.checkArgument((!singleResourceSpec.endsWith("/") ? 1 : 0) != 0, (String)"Expected a file path, but [%s], ends with '/'. This is unsupported in GcsFileSystem.", (Object)singleResourceSpec);
        }
        GcsPath path = GcsPath.fromUri(singleResourceSpec);
        return GcsResourceId.fromGcsPath(path);
    }

    protected void copy(List<GcsResourceId> srcResourceIds, List<GcsResourceId> destResourceIds) throws IOException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        this.options.getGcsUtil().copy(this.toFilenames(srcResourceIds), this.toFilenames(destResourceIds));
        stopwatch.stop();
        if (this.options.getGcsPerformanceMetrics().booleanValue()) {
            this.numCopies.inc((long)srcResourceIds.size());
            this.copyTimeMsec.inc(stopwatch.elapsed(TimeUnit.MILLISECONDS));
        }
    }

    protected String getScheme() {
        return "gs";
    }

    private List<MatchResult> matchGlobs(List<GcsPath> globs) {
        return FluentIterable.from(globs).transform(gcsPath -> {
            try {
                return this.expand((GcsPath)gcsPath);
            }
            catch (IOException e) {
                return MatchResult.create((MatchResult.Status)MatchResult.Status.ERROR, (IOException)e);
            }
        }).toList();
    }

    @VisibleForTesting
    MatchResult expand(GcsPath gcsPattern) throws IOException {
        Objects objects;
        String prefix = GcsUtil.getNonWildcardPrefix(gcsPattern.getObject());
        Pattern p = Pattern.compile(FileSystemUtils.wildcardToRegexp((String)gcsPattern.getObject()));
        LOG.debug("matching files in bucket {}, prefix {} against pattern {}", new Object[]{gcsPattern.getBucket(), prefix, p.toString()});
        String pageToken = null;
        ArrayList<MatchResult.Metadata> results = new ArrayList<MatchResult.Metadata>();
        while ((objects = this.options.getGcsUtil().listObjects(gcsPattern.getBucket(), prefix, pageToken)).getItems() != null) {
            for (StorageObject o : objects.getItems()) {
                String name = o.getName();
                if (!p.matcher(name).matches() || name.endsWith("/")) continue;
                LOG.debug("Matched object: {}", (Object)name);
                results.add(this.toMetadata(o));
            }
            pageToken = objects.getNextPageToken();
            if (pageToken != null) continue;
        }
        return MatchResult.create((MatchResult.Status)MatchResult.Status.OK, results);
    }

    @VisibleForTesting
    List<MatchResult> matchNonGlobs(List<GcsPath> gcsPaths) throws IOException {
        List<GcsUtil.StorageObjectOrIOException> results = this.options.getGcsUtil().getObjects(gcsPaths);
        ImmutableList.Builder ret = ImmutableList.builder();
        for (GcsUtil.StorageObjectOrIOException result : results) {
            ret.add((Object)this.toMatchResult(result));
        }
        return ret.build();
    }

    private MatchResult toMatchResult(GcsUtil.StorageObjectOrIOException objectOrException) {
        @Nullable IOException exception = objectOrException.ioException();
        if (exception instanceof FileNotFoundException) {
            return MatchResult.create((MatchResult.Status)MatchResult.Status.NOT_FOUND, (IOException)exception);
        }
        if (exception != null) {
            return MatchResult.create((MatchResult.Status)MatchResult.Status.ERROR, (IOException)exception);
        }
        StorageObject object = objectOrException.storageObject();
        assert (object != null);
        return MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)this.toMetadata(object)));
    }

    private MatchResult.Metadata toMetadata(StorageObject storageObject) {
        MatchResult.Metadata.Builder ret = MatchResult.Metadata.builder().setIsReadSeekEfficient(true).setResourceId((ResourceId)GcsResourceId.fromGcsPath(GcsPath.fromObject(storageObject)));
        if (storageObject.getMd5Hash() != null) {
            ret.setChecksum(storageObject.getMd5Hash());
        }
        BigInteger size = (BigInteger)MoreObjects.firstNonNull((Object)storageObject.getSize(), (Object)BigInteger.ZERO);
        ret.setSizeBytes(size.longValue());
        DateTime lastModified = (DateTime)MoreObjects.firstNonNull((Object)storageObject.getUpdated(), (Object)new DateTime(0L));
        ret.setLastModifiedMillis(lastModified.getValue());
        return ret.build();
    }

    private List<String> toFilenames(Collection<GcsResourceId> resources) {
        return FluentIterable.from(resources).transform(resource -> resource.getGcsPath().toString()).toList();
    }

    private List<GcsPath> toGcsPaths(Collection<String> specs) {
        return FluentIterable.from(specs).transform(GcsPath::fromUri).toList();
    }
}

