/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.ComponentModel;

using PostgreSql.Data.PgTypes;

namespace PostgreSql.Data.PgSqlClient
{
	[ParenthesizePropertyName(true),
	TypeConverter(typeof(Design.PgParameterConverter))]
	public sealed class PgParameter : MarshalByRefObject, IDbDataParameter, IDataParameter, ICloneable
	{
		#region Fields
				
		ParameterDirection	direction;
		DataRowVersion		sourceVersion;
		bool				isNullable;
		string				parameterName;
		string				sourceColumn;
		object				value;
		byte				precision;
		byte				scale;
		int					size;
		PgDbType			pgDbType;
		bool				defaultCtor;

		#endregion

		#region Properties

		string IDataParameter.ParameterName
		{
			get { return ParameterName; }
			set { ParameterName = value; }
		}

		[DefaultValue("")]
		public string ParameterName 
		{
			get { return parameterName; }
			set { parameterName = value; }
		}

		[Category("Data"), DefaultValue((byte)0)]
		public byte Precision
		{
			get { return precision; }
			set { precision = value; }
		}

		[Category("Data"), DefaultValue((byte)0)]
		public byte Scale
		{
			get { return scale; }
			set { scale = value; }
		}

		[Category("Data"), DefaultValue(0)]
		public int Size
		{
			get { return size; }
			set { size = value; }
		}

		[Browsable(false),
		Category("Data"),
		RefreshProperties(RefreshProperties.All),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public DbType DbType 
		{
			get { return PgDbTypeToDbType(pgDbType); }
			set { pgDbType = dbTypeToPgType(value); }
		}

		[RefreshProperties(RefreshProperties.All),
		Category("Data"),
		DefaultValue(PgDbType.VarChar)]
		public PgDbType PgDbType
		{
			get { return pgDbType; }
			set { pgDbType = value; }
		}

		[Category("Data"), DefaultValue(ParameterDirection.Input)]
		public ParameterDirection Direction 
		{
			get { return direction; }
			set { direction = value; }
		}

		[Browsable(false), 
		DesignOnly(true), 
		DefaultValue(false),
		EditorBrowsable(EditorBrowsableState.Advanced)]
		public bool IsNullable
		{
			get { return isNullable; }
			set { isNullable = value; }  
		}

		[Category("Data"), DefaultValue("")]
		public string SourceColumn
		{
			get { return sourceColumn; }
			set { sourceColumn = value; }
		}

		[Category("Data"), DefaultValue(DataRowVersion.Current)]
		public DataRowVersion SourceVersion 
		{
			get { return sourceVersion; }
			set { sourceVersion = value; }
		}

		[Category("Data"), 
		TypeConverter(typeof(StringConverter)),
		DefaultValue(null)]
		public object Value 
		{
			get { return this.value; }
			set
			{
				if (value == null)
				{
					value = System.DBNull.Value;
				}
				this.value = value;
				if (defaultCtor)
				{
					setPgTypeFromValue(this.value);
				}
			}
		}

		#endregion

		#region Constructors

		public PgParameter()
		{
			this.direction		= ParameterDirection.Input;
			this.sourceVersion	= DataRowVersion.Current;
			this.isNullable		= false;
			this.pgDbType		= PgDbType.VarChar;
			this.defaultCtor	= true;
		}

		public PgParameter(string parameterName, object value)  : this()
		{
			this.defaultCtor	= false;
			this.parameterName	= parameterName;
			this.value			= value;
		}

		public PgParameter(string parameterName, PgDbType dbType) : this()
		{
			this.defaultCtor	= false;
			this.parameterName	= parameterName;
			this.pgDbType		= dbType;
		}

		public PgParameter(string parameterName, PgDbType dbType, int size) : this()
		{
			this.defaultCtor	= false;
			this.parameterName	= parameterName;
			this.pgDbType		= dbType;
			this.size			= size;
		}

		public PgParameter(string parameterName, PgDbType dbType, int size, string sourceColumn) : this()
		{
			this.defaultCtor	= false;
			this.parameterName	= parameterName;
			this.pgDbType		= dbType;
			this.size			= size;
			this.sourceColumn	= sourceColumn;
		}

		[EditorBrowsable(EditorBrowsableState.Advanced)]
		public PgParameter(string parameterName,
							PgDbType dbType,
							int size,
							ParameterDirection direction,
							bool isNullable,
							byte precision,
							byte scale,
							string sourceColumn,
							DataRowVersion sourceVersion,
							object value)
		{
			this.defaultCtor	= false;
			this.parameterName	= parameterName;
			this.pgDbType		= dbType;
			this.size			= size;
			this.direction		= direction;
			this.isNullable		= isNullable;
			this.precision		= precision;
			this.scale			= scale;
			this.sourceColumn	= sourceColumn;
			this.sourceVersion	= sourceVersion;
			this.value			= value;
		}

		#endregion

		#region ICloneable Method

		object ICloneable.Clone()
		{
			return new PgParameter(parameterName,
							PgDbType,
							size,
							direction,
							isNullable,
							precision,
							scale,
							sourceColumn,
							sourceVersion,
							value);
		}

		#endregion

		#region Methods

		public override string ToString()
		{
			return this.parameterName;
		}
	
		#endregion

		#region Internal Methods

		internal string ConvertToPgString()
		{
			bool	addQuotes	= true;
			string	returnValue = String.Empty;

			switch (this.pgDbType)
			{
				case PgDbType.Array:
					break;

				case PgDbType.Binary:
					break;

				case PgDbType.Boolean:
					returnValue = Convert.ToBoolean(this.value).ToString().ToLower();
					break;

				case PgDbType.Box:
					returnValue = ((PgBox)this.value).ToString();
					break;

				case PgDbType.Byte:
					returnValue = Convert.ToByte(this.value).ToString();
					break;

				case PgDbType.Char:
				case PgDbType.VarChar:
				case PgDbType.Text:
					returnValue = Convert.ToString(this.value);
					break;

				case PgDbType.Circle:
					returnValue = ((PgCircle)this.value).ToString();
					break;

				case PgDbType.Currency:					
					returnValue = "$" + Convert.ToSingle(this.value).ToString();
					break;

				case PgDbType.Date:
					returnValue = Convert.ToDateTime(this.value).ToString("MM/dd/yyyy");
					break;

				case PgDbType.Decimal:
				case PgDbType.Numeric:
					returnValue = Convert.ToDecimal(this.value).ToString();
					break;

				case PgDbType.Double:
					returnValue = Convert.ToDouble(this.value).ToString();
					break;

				case PgDbType.Float:
					returnValue = Convert.ToSingle(this.value).ToString();
					break;

				case PgDbType.Int2:
					returnValue = Convert.ToInt16(this.value).ToString();
					break;

				case PgDbType.Int4:
					returnValue = Convert.ToInt32(this.value).ToString();
					break;

				case PgDbType.Int8:
					returnValue = Convert.ToInt64(this.value).ToString();
					break;

				case PgDbType.Interval:
					break;

				case PgDbType.Line:
					returnValue = ((PgLine)this.value).ToString();
					break;

				case PgDbType.LSeg:
					returnValue = ((PgLSeg)this.value).ToString();
					break;
				
				case PgDbType.Path:
					returnValue = ((PgPath)this.value).ToString();
					break;

				case PgDbType.Point:
					returnValue = ((PgPoint)this.value).ToString();
					break;

				case PgDbType.Polygon:
					returnValue = ((PgPolygon)this.value).ToString();
					break;
				
				case PgDbType.Time:
					returnValue = Convert.ToDateTime(this.value).ToString("HH:mm:ss");
					break;

				case PgDbType.Timestamp:
					returnValue = Convert.ToDateTime(this.value).ToString("MM/dd/yyy HH:mm:ss");
					break;

				case PgDbType.TimestampWithTZ:
					returnValue = Convert.ToDateTime(this.value).ToString("MM/dd/yyy HH:mm:ss zz");
					break;

				case PgDbType.TimeWithTZ:
					returnValue = Convert.ToDateTime(this.value).ToString("HH:mm:ss zz");
					break;

				case PgDbType.Vector:
					break;

				default:
					returnValue = this.value.ToString();
					break;
			}

			if (addQuotes)
			{
				returnValue = "'" + returnValue + "'";
			}

			return returnValue;
		}

		#endregion

		#region Private Methods

		private void setPgTypeFromValue(object value)
		{
			if (value == null)
			{
				value = System.DBNull.Value;
			}

			switch (Type.GetTypeCode(value.GetType()))
			{
				case TypeCode.Byte:
					PgDbType = PgDbType.Byte;
					break;

				case TypeCode.Boolean:
					PgDbType = PgDbType.Boolean;
					break;

				case TypeCode.Object:
					PgDbType = PgDbType.Binary;
					break;
			
				case TypeCode.String:
				case TypeCode.Char:
					PgDbType = PgDbType.Char;
					break;

				case TypeCode.Int16:
					PgDbType = PgDbType.Int2;
					break;

				case TypeCode.Int32:
					PgDbType = PgDbType.Int4;
					break;

				case TypeCode.Int64:
					PgDbType = PgDbType.Int8;
					break;

				case TypeCode.Single:
					PgDbType = PgDbType.Float;
					break;

				case TypeCode.Double:
					PgDbType = PgDbType.Double;
					break;

				case TypeCode.Decimal:
					PgDbType = PgDbType.Decimal;
					break;

				case TypeCode.DateTime:
					PgDbType = PgDbType.Timestamp;
					break;

				case TypeCode.DBNull:
					break;

				case TypeCode.Empty:
				case TypeCode.SByte:
				case TypeCode.UInt16:
				case TypeCode.UInt32:
				case TypeCode.UInt64:
				default:
					throw new SystemException("Value is of invalid data type.");
			}
		}

		private DbType PgDbTypeToDbType(PgDbType PgDbType)
		{
			switch(PgDbType)
			{
				case PgDbType.Boolean:
					return DbType.Boolean;

				case PgDbType.Byte:
					return DbType.Byte;

				case PgDbType.Binary:
					return DbType.Binary;

				case PgDbType.Char:
				case PgDbType.Text:				
				case PgDbType.VarChar:
					return DbType.String;

				case PgDbType.Int2:
					return DbType.Int16;

				case PgDbType.Int4:
					return DbType.Int32;

				case PgDbType.Int8:
					return DbType.Int64;

				case PgDbType.Date:
					return DbType.Date;

				case PgDbType.Time:
					return DbType.Time;

				case PgDbType.Timestamp:
					return DbType.DateTime;

				case PgDbType.Decimal:
				case PgDbType.Numeric:
					return DbType.Decimal;
			
				case PgDbType.Float:
					return DbType.Single;

				case PgDbType.Double:
					return DbType.Double;

				default:
					throw new InvalidOperationException("Invalid data type specified.");
			}			
		}

		private PgDbType dbTypeToPgType(DbType dbType)
		{			
			switch(dbType)
			{
				case DbType.Byte:
					return PgDbType.Byte;

				case DbType.Boolean:
					return PgDbType.Boolean;									

				case DbType.AnsiString:
				case DbType.String:
					return PgDbType.VarChar;
				
				case DbType.AnsiStringFixedLength:
				case DbType.StringFixedLength:
					return PgDbType.Char;
				
				case DbType.Binary:
				case DbType.Object:
					return PgDbType.Binary;
				
				case DbType.Date:
					return PgDbType.Date;

				case DbType.Time:
					return PgDbType.Time;

				case DbType.DateTime:
					return PgDbType.Timestamp;

				case DbType.Int16:
				case DbType.UInt16:
					return PgDbType.Int2;

				case DbType.Int32:
				case DbType.UInt32:
					return PgDbType.Int4;

				case DbType.Int64:
				case DbType.UInt64:
					return PgDbType.Int8;

				case DbType.Single:
					return PgDbType.Float;

				case DbType.Decimal:
					return PgDbType.Decimal;

				case DbType.Double:
					return PgDbType.Double;

				case DbType.Currency:
					return PgDbType.Currency;

				case DbType.Guid:
				case DbType.VarNumeric:
				case DbType.SByte:
				default:
					throw new InvalidOperationException("Invalid data type specified.");
			}
		}

		#endregion
	}
}