/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Net;
using System.IO;
using System.Text;
using System.Globalization;

using PostgreSql.Data.PgTypes;

namespace PostgreSql.Data.NPgClient
{
	internal class PgOutputPacket : BinaryWriter
	{
		#region Fields

		private Encoding encoding;

		#endregion

		#region Properties

		public long Position
		{
			get { return ((MemoryStream)BaseStream).Position; }
		}

		public long Length
		{
			get { return ((MemoryStream)BaseStream).Length; }
		}

		#endregion

		#region Constructors

		public PgOutputPacket() : base(new MemoryStream())
		{
			this.encoding = Encoding.Default;
			Write(new byte[0]);
		}

		public PgOutputPacket(Encoding encoding) : base(new MemoryStream(), encoding)
		{
			this.encoding = encoding;
			this.Write(new byte[0]);
		}

		#endregion

		#region String Types

		public void WriteString(string data)
		{
			if (!data.EndsWith(PgCodes.NULL_TERMINATOR.ToString()))
			{
				data += PgCodes.NULL_TERMINATOR;
			}
			this.Write(encoding.GetBytes(data));
		}

		#endregion

		#region Numeric Types

		public void WriteShort(short val)
		{
			this.Write((short)IPAddress.HostToNetworkOrder(val));
		}

		public void WriteInt(int val)
		{
			this.Write((int)IPAddress.HostToNetworkOrder(val));
		}

		public void WriteLong(long val)
		{
			this.Write((long)IPAddress.HostToNetworkOrder(val));
		}

		public void WriteFloat(float val)
		{
			FloatLayout floatValue = new FloatLayout();			

			floatValue.f = val;
			floatValue.i0 = IPAddress.HostToNetworkOrder(floatValue.i0);

			this.Write(floatValue.f);
		}

		public void WriteDouble(double val)
		{
			DoubleLayout doubleValue = new DoubleLayout();
			int temp;			

			doubleValue.d = val;
			doubleValue.i0 = IPAddress.HostToNetworkOrder(doubleValue.i0);
			doubleValue.i4 = IPAddress.HostToNetworkOrder(doubleValue.i4);

			temp = doubleValue.i0;
			doubleValue.i0 = doubleValue.i4;
			doubleValue.i4 = temp;

			this.Write(doubleValue.d);
		}

		#endregion

		#region Date & Time Types

		public void WriteDate(DateTime date)
		{
			TimeSpan days = date.Subtract(PgCodes.BASE_DATE);

			this.WriteInt(days.Days);
		}

		public void WriteInterval(TimeSpan interval)
		{
			int months	= (interval.Days / 30);
			int days	= (interval.Days % 30);

			this.WriteDouble(interval.Subtract(TimeSpan.FromDays(months * 30)).TotalSeconds);
			this.WriteInt(months);
		}

		public void WriteTime(DateTime time)
		{
			DateTime realTime = new DateTime(PgCodes.BASE_DATE.Year,
				PgCodes.BASE_DATE.Month,
				PgCodes.BASE_DATE.Day,
				time.Hour,
				time.Minute,
				time.Second,
				time.Millisecond);

			TimeSpan seconds = realTime.Subtract(PgCodes.BASE_DATE);

			this.WriteDouble(seconds.TotalSeconds);
		}

		public void WriteTimeWithTZ(DateTime time)
		{
			DateTime realTime = new DateTime(PgCodes.BASE_DATE.Year,
				PgCodes.BASE_DATE.Month,
				PgCodes.BASE_DATE.Day,
				time.Hour,
				time.Minute,
				time.Second,
				time.Millisecond);

			TimeSpan seconds = realTime.Subtract(PgCodes.BASE_DATE);

			this.WriteDouble(seconds.TotalSeconds);
			this.WriteInt((-1)*Int32.Parse(time.ToString("zz"))*3600);
		}

		public void WriteTimestamp(DateTime timestamp)
		{
			TimeSpan days = timestamp.Subtract(PgCodes.BASE_DATE);

			this.WriteDouble(days.TotalSeconds);
		}

		public void WriteTimestampWithTZ(DateTime timestamp)
		{
			this.WriteTimestamp(timestamp);
		}

		#endregion

		#region Geometric Types

		public void WritePoint(PgPoint point)
		{
			this.WriteDouble(point.X);
			this.WriteDouble(point.Y);
		}

		public void WriteCircle(PgCircle circle)
		{
			this.WritePoint(circle.Center);
			this.WriteDouble(circle.Radius);
		}

		public void WriteLine(PgLine line)
		{
			this.WritePoint(line.StartPoint);
			this.WritePoint(line.EndPoint);
		}

		public void WriteLSeg(PgLSeg lseg)
		{
			this.WritePoint(lseg.StartPoint);
			this.WritePoint(lseg.EndPoint);
		}

		public void WriteBox(PgBox box)
		{
			this.WritePoint(box.UpperRight);
			this.WritePoint(box.LowerLeft);
		}

		public void WritePolygon(PgPolygon polygon)
		{
			this.WriteInt(polygon.Points.Length);
			for (int i = 0; i < polygon.Points.Length; i++)
			{
				this.WritePoint(polygon.Points[i]);
			}
		}

		public void WritePath(PgPath path)
		{
			this.Write(path.IsClosedPath);
			this.WriteInt(path.Points.Length);
			for (int i = 0; i < path.Points.Length; i++)
			{
				this.WritePoint(path.Points[i]);
			}
		}

		#endregion

		#region Parameters

		public void WriteParameter(PgParameter parameter)
		{
			int size = parameter.DataType.Size;

			if (parameter.Value == System.DBNull.Value || 
				parameter.Value == null)
			{
				// -1 indicates a NULL argument value
				this.WriteInt(-1);
			}
			else
			{
				if (parameter.DataType.DataType == PgDataType.Binary ||
					parameter.DataType.DataType == PgDataType.Array	||
					parameter.DataType.DataType == PgDataType.Vector)
				{
					// Handle this type as Array values
					System.Array array = (System.Array)parameter.Value;
					
					// Get array elements type info
					PgType elementType = PgDbClient.Types[parameter.DataType.ElementType];
					size = elementType.Size;

					// Create a new packet for write array parameter information
					PgOutputPacket packet = new PgOutputPacket();

					// Write the number of dimensions
					packet.WriteInt(array.Rank);

					// Write flags (always 0)
					packet.WriteInt(0);

					// Write base type of the array elements
					packet.WriteInt(parameter.DataType.ElementType);

					// Write lengths and lower bounds 
					for (int i = 0; i < array.Rank; i ++)
					{
						packet.WriteInt(array.GetLength(i));
						packet.WriteInt(array.GetLowerBound(i) + 1);
					}

					// Write array values
					foreach (object element in array)
					{
						this.writeParameter(packet, elementType.DataType, size, element);
					}

					// Write parameter size
					this.WriteInt(packet.GetByteCount());
					// Write parameter data
					this.Write(packet.ToArray());
				}
				else
				{
					this.writeParameter(this, parameter.DataType.DataType, size, parameter.Value);
				}
			}
		}

		#endregion

		#region Private Methods

		private void writeParameter(PgOutputPacket packet, PgDataType dataType, int size, object value)
		{
			switch (dataType)
			{
				case PgDataType.Binary:
					packet.WriteInt(((byte[])value).Length);
					packet.Write((byte[])value);
					break;

				case PgDataType.Byte:
					packet.WriteInt(size);
					packet.Write((byte)value);
					break;

				case PgDataType.Int2:
					packet.WriteInt(size);
					packet.WriteShort(Convert.ToInt16(value));
					break;

				case PgDataType.Int4:
					packet.WriteInt(size);
					packet.WriteInt(Convert.ToInt32(value));
					break;

				case PgDataType.Int8:
					packet.WriteInt(size);
					packet.WriteLong(Convert.ToInt64(value));
					break;

				case PgDataType.Interval:
					packet.WriteInt(size);
					packet.WriteInterval(TimeSpan.Parse(value.ToString()));
					break;

				case PgDataType.Decimal:
				{
					string paramValue = value.ToString();
					packet.WriteInt(encoding.GetByteCount(paramValue));
					packet.Write(paramValue.ToCharArray());
				}
				break;

				case PgDataType.Double:
					packet.WriteInt(size);
					packet.WriteDouble(Convert.ToDouble(value));
					break;
				
				case PgDataType.Float:
					packet.WriteInt(size);
					packet.WriteFloat(Convert.ToSingle(value));
					break;

				case PgDataType.Currency:
					packet.WriteInt(size);
					packet.WriteInt(Convert.ToInt32(Convert.ToSingle(value)*100));
					break;

				case PgDataType.Date:
					packet.WriteInt(size);
					packet.WriteDate(Convert.ToDateTime(value));
					break;

				case PgDataType.Time:
					packet.WriteInt(size);
					packet.WriteTime(Convert.ToDateTime(value));
					break;

				case PgDataType.TimeWithTZ:
					packet.WriteInt(size);
					packet.WriteTimeWithTZ(Convert.ToDateTime(value));
					break;

				case PgDataType.Timestamp:
					packet.WriteInt(size);
					packet.WriteTimestamp(Convert.ToDateTime(value));
					break;

				case PgDataType.TimestampWithTZ:
					packet.WriteInt(size);
					packet.WriteTimestampWithTZ(Convert.ToDateTime(value));
					break;

				case PgDataType.Char:
				case PgDataType.VarChar:
				{
					string paramValue = value.ToString() + PgCodes.NULL_TERMINATOR; 
					packet.WriteInt(encoding.GetByteCount(paramValue));
					packet.WriteString(paramValue);
				}
				break;

				case PgDataType.Point:
					packet.WriteInt(size);
					packet.WritePoint((PgPoint)value);
					break;
				
				case PgDataType.Circle:
					packet.WriteInt(size);
					packet.WriteCircle((PgCircle)value);
					break;

				case PgDataType.Line:
					packet.WriteInt(size);
					packet.WriteLine((PgLine)value);
					break;

				case PgDataType.LSeg:
					packet.WriteInt(size);
					packet.WriteLSeg((PgLSeg)value);
					break;

				case PgDataType.Box:
					packet.WriteInt(size);
					packet.WriteBox((PgBox)value);
					break;

				case PgDataType.Polygon:
					PgPolygon polygon = (PgPolygon)value;
					
					packet.WriteInt((size*polygon.Points.Length) + 4);
					packet.WritePolygon(polygon);
					break;

				case PgDataType.Path:
					PgPath path = (PgPath)value;

					packet.WriteInt((size*path.Points.Length) + 5);
					packet.WritePath(path);
					break;
			}
		}

		#endregion

		#region Stream Methods

		public int GetByteCount()
		{
			return (int)((MemoryStream)BaseStream).Length;
		}

		public byte[] ToArray()
		{
			return ((MemoryStream)BaseStream).ToArray();
		}

		public void Reset()
		{
			((MemoryStream)BaseStream).SetLength(0);
			((MemoryStream)BaseStream).Position = 0;
		}

		#endregion

		#region Packet Methods

		public byte[] GetSimplePacketBytes()
		{
			PgOutputPacket packet = new PgOutputPacket();

			// Write packet contents
			packet.WriteInt(GetByteCount() + 4);
			packet.Write(ToArray());

			return packet.ToArray();
		}

		public byte[] GetPacketBytes(char format)
		{
			PgOutputPacket packet = new PgOutputPacket();

			packet.Write((byte)format);
			packet.WriteInt(GetByteCount() + 4);
			packet.Write(ToArray());

			return packet.ToArray();
		}

		#endregion
	}
}