package org.springframework.uaa.client.internal;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.xml.parsers.ParserConfigurationException;

import org.springframework.uaa.client.UaaDetectedProducts;
import org.springframework.uaa.client.util.Base64;
import org.springframework.uaa.client.util.PreferencesUtils;
import org.springframework.uaa.client.util.StreamUtils;
import org.springframework.uaa.client.util.XmlUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Represents common product names and dependency information.
 * 
 * <p>
 * Spring UAA allows product names to be expressed as a string. This freedom is necessary as new products
 * may be added at any time and we shouldn't need to publish a new version of UAA just to support them.
 * However, to ensure a reasonable level of product name consistency for backend correlation, this class
 * defines official product names. It also offers Maven IDs used by open source products to facilitate
 * automatic detection by some UAA-aware tools that can iterate user project classpaths.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * @since 1.0
 *
 */
public final class UaaDetectedProductsImpl implements UaaDetectedProducts {
	private static final String DETECTED_PRODUCTS_KEY = "detected_products";
	private static final String EMPTY_STRING = "";
	private static final byte[] EMPTY = {};
	private static final Preferences P = PreferencesUtils.getPreferencesFor(UaaDetectedProductsImpl.class);
	private static final List<ProductInfo> products = new ArrayList<ProductInfo>();
	private static int version = 0;
	
	public ProductInfo getDetectedProductInfo(String groupId, String artifactId) {
		if (groupId != null && artifactId != null) {
			for (ProductInfo productInfo : getDetectedProductInfos()) {
				if (productInfo.getGroupId().equals(groupId) && productInfo.getArtifactId().equals(artifactId)) {
					return productInfo;
				}
			}
		}
		return null;
	}
	
	/**
	 * @return an unmodifiable representation of all products declared by this class (never returns null)
	 */
	public List<ProductInfo> getDetectedProductInfos() {
		initializeIfRequired();
		return Collections.unmodifiableList(products);

	}
	
	public boolean shouldReportUsage(String groupId, String artifactId) {
		return getDetectedProductInfo(groupId, artifactId) != null;
	}
	
	/**
	 * @return the version of the detected products XML document
	 */
	public static final int getVersion() {
		initializeIfRequired();
		return version;
	}
	
	/**
	 * Allows a tool to update the list of known products. The updated list of known products is typically
	 * obtained via HTTP from the {@link #PRODUCT_URL}. Of course this download should only happen if the user
	 * has provided consent at a privacy level.
	 * 
	 * <p>
	 * The existing product list will be retained if the incoming stream is null, invalid XML, missing the expected
	 * XML elements, or is an older version than the existing product list. The new product list will be cached in the
	 * user preferences to avoid external tools needing to do this.
	 * 
	 * <p>
	 * This method never throws any exceptions. Even runtime exceptions are caught.
	 * 
	 * @param inputXmlDocument the replacement XML document (can be null)
	 * @return true if the incoming document was stored successfully and replaced any earlier version, false otherwise
	 */
	static boolean setProducts(InputStream inputXmlDocument) {
		if (inputXmlDocument == null) return false;
		try {
			// Convert the incoming input stream into a byte[] to start with (simplifies subsequent storage etc)
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			StreamUtils.copy(inputXmlDocument, baos);
			byte[] incoming = baos.toByteArray();
			if (incoming.length == 0) return false;
			
			// Retrieve the existing document from the Preference API (if any) so we can consider version upgrade issues
			byte[] existing = Base64.decode(P.get(DETECTED_PRODUCTS_KEY, EMPTY_STRING));
			if (existing != EMPTY && existing.length > 0) {
				// We need to consider version differences
				int existingVersion = getVersion(existing);
				int incomingVersion = getVersion(incoming);
				if (existingVersion >= incomingVersion) {
					// We're not going to do any replacement as the new version is no better than our current version
					// Instead we'll treat our existing version as the newest version and continue running
					incoming = existing;
				}
			}
			
			// Parse the incoming bytes into an XML document
			version = getVersion(incoming);
			Document d = XmlUtils.parse(new ByteArrayInputStream(incoming));
			Element docElement = d.getDocumentElement();

			// Build products list
			NodeList nodeList = docElement.getElementsByTagName("product");
			products.clear();
			for (int i = 0; i < nodeList.getLength(); i++) {
				Node n = nodeList.item(i);
				if (n instanceof Element) {
					Element product = (Element) n;
					String productName = product.getAttribute("name");
					String groupId = product.getAttribute("group-id");
					String artifactId = product.getAttribute("artifact-id");
					String typeName = product.getAttribute("type-name");
					if (productName.length() > 0 && groupId.length() > 0 && artifactId.length() > 0) {
						products.add(new ProductInfo(productName, groupId, artifactId, typeName));
					}
				}
			}
			
			// Replace the existing cached version if required
			if (incoming != existing) {
				// We need to do a replacement, as we didn't simply parse the existing one
				String base64 = Base64.encodeBytes(incoming, Base64.GZIP);
				P.put(DETECTED_PRODUCTS_KEY, base64);
				return true;
			}
			
			return false;
			
		} catch (Throwable ignore) {
			return false;
		}
	}
	
	/**
	 * Looks up the version attribute from the incoming XML document's <product version='someInteger'>.
	 * 
	 * @param document to parse (required)
	 * @return the version number (or an exception of something went wrong, eg invalid document etc)
	 */
	private static int getVersion(byte[] document) throws ParserConfigurationException, IOException, SAXException {
		Document d = XmlUtils.parse(new ByteArrayInputStream(document));
		Element docElement = d.getDocumentElement();
		String ver = docElement.getAttribute("version");
		return  new Integer(ver);
	}

	private static void initializeIfRequired() {
		if (products.size() == 0 || version == 0) {
			// We always start by presenting our classpath-provided DetectedProducts.xml 
			// (the setter handles storage via Preferences API, best version resolution etc)
			InputStream defaults = getDetectedProductsFromClassLoader();
			setProducts(defaults);
		}
	}
	
	/**
	 * Used by tests to delete all the keys associated with this class from the Preferences API.
	 */
	static void clearAllKeys() {
		try {
			P.clear();
		} catch (BackingStoreException e) {
			throw new IllegalStateException(e);
		}
	}

	static InputStream getDetectedProductsFromClassLoader() {
		return UaaDetectedProductsImpl.class.getResourceAsStream("/org/springframework/uaa/client/uaa-client.xml");
	}

	/**
	 * Used by tests to keep track of how many keys are in use.
	 * 
	 * @return the number of preferences API keys used by this class (should be 1, ie DETECTED_PRODUCTS_KEY)
	 */
	static int getPreferencesKeyCount() {
		try {
			return P.keys().length;
		} catch (BackingStoreException e) {
			throw new IllegalStateException(e);
		}
	}

	/**
	 * Used by tests. Resets the detected products to the default that ships with this version of UAA. This intentionally discards
	 * any possibly-newer version obtained via the preferences API.
	 */
	static void resetProducts() {
		P.remove(DETECTED_PRODUCTS_KEY);
		InputStream defaults = UaaDetectedProductsImpl.class.getResourceAsStream("/org/springframework/uaa/client/uaa-client.xml");
		setProducts(defaults);
	}
	
	
}

