package org.springframework.uaa.client;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.Manifest;

import org.springframework.uaa.client.protobuf.UaaClient.FeatureUse;
import org.springframework.uaa.client.protobuf.UaaClient.Product;
import org.springframework.uaa.client.util.Assert;

/**
 * Helps parse version numbers from system properties, manifests and arbitrary strings.
 * 
 * @author Ben Alex
 *
 */
public abstract class VersionHelper {
	private static final String javaRuntime = System.getProperty("java.runtime.name");
	private static final String javaVendor = System.getProperty("java.vendor");
	private static final String javaVersion = System.getProperty("java.runtime.version");
	private static final String osName = System.getProperty("os.name");
	private static final String osArch= System.getProperty("os.arch");
	private static final String osVersion = System.getProperty("os.version");
	
	/**
	 * Provides the product name, version and Git commit ID in an easily-displayed form.
	 * 
	 * @param uaa the product to return the information for (required)
	 * @return a string representing the product (never null or empty)
	 */
	public static String getProductStamp(Product uaa) {
		String source = "";
		String id = uaa.getSourceControlIdentifier();
		if (id != null && !"".equals(id)) {
			if (id.length() > 7) {
				id = id.substring(0,7);
			}
			source = "[rev " + id + "]";
		}
		return uaa.getName() + " " + uaa.getMajorVersion() + "." + uaa.getMinorVersion() + "." + uaa.getPatchVersion() + "." + uaa.getReleaseQualifier() + " " + source;
	}
	
	public static Product getJvm() {
		Product.Builder b = getProductBuilder(javaVersion);
		b.setName(javaVendor + " " + javaRuntime);
		return b.build();
	}
	
	public static Product getOs() {
		Product.Builder b = getProductBuilder(osVersion);
		b.setName(osName + " " + osArch);
		return b.build();
	}

	public static Product getUaa() {
		try {
			return getProductFromManifest(VersionHelper.class, UaaDetectedProducts.SPRING_UAA.getProductName());
		} catch (Exception e) {
			// Manifest parse failed; probably being run outside a JAR (eg development mode etc)
			return getProduct(UaaDetectedProducts.SPRING_UAA.getProductName(), "0.0.0.MILESTONE");
		}
	}

	// package protected for unit tests
	static Product.Builder getProductBuilder(String versionString) {
		Product.Builder b = Product.newBuilder();
		GetNumbersResult getNumbersResult = getNumbers(versionString, 3); // we only need the first 3 numbers
		if (getNumbersResult.numbers.size() >= 1) b.setMajorVersion(getNumbersResult.numbers.get(0));
		if (getNumbersResult.numbers.size() >= 2) b.setMinorVersion(getNumbersResult.numbers.get(1));
		if (getNumbersResult.numbers.size() >= 3) b.setPatchVersion(getNumbersResult.numbers.get(2));
		b.setReleaseQualifier(getNumbersResult.restOfString);
		return b;
	}

	// package protected for unit tests
	static FeatureUse.Builder getFeatureUseBuilder(String versionString) {
		FeatureUse.Builder b = FeatureUse.newBuilder();
		if (versionString != null) {
			GetNumbersResult getNumbersResult = getNumbers(versionString, 3); // we only need the first 3 numbers
			if (getNumbersResult.numbers.size() >= 1) b.setMajorVersion(getNumbersResult.numbers.get(0));
			if (getNumbersResult.numbers.size() >= 2) b.setMinorVersion(getNumbersResult.numbers.get(1));
			if (getNumbersResult.numbers.size() >= 3) b.setPatchVersion(getNumbersResult.numbers.get(2));
			b.setReleaseQualifier(getNumbersResult.restOfString);
		}
		return b;
	}
	
	/**
	 * Delegates to {@link #getFeatureUse(String, String, String)}, passing null as the simpleVersionSequence 
	 * and sourceId.
	 * 
	 * @param featureName the product name (required)
	 * @return a {@link FeatureUse} assuming the item could be parsed (never null)
	 */
	public static FeatureUse getFeatureUse(String featureName) {
		return getFeatureUse(featureName, null, null);
	}

	/**
	 * Delegates to {@link #getFeatureUse(String, String, String)}, passing null as the sourceId.
	 * 
	 * @param featureName the product name (required)
	 * @param simpleVersionSequence a version number such as ("1.0.4.M101"; null is acceptable)
	 * @return a {@link FeatureUse} assuming the item could be parsed (never null)
	 */
	public static FeatureUse getFeatureUse(String featureName, String simpleVersionSequence) {
		return getFeatureUse(featureName, simpleVersionSequence, null);
	}
	
	/**
	 * Useful for products to obtain a {@link FeatureUse} without using the builder. This method can be
	 * used with any version number adopting the form <code>I.I.I.CI</code> where "I" means an
	 * integer, "C" means one or more characters to identify the release category, followed by an
	 * integer-based release qualifier. For example, "1.42.5.M1" or "5.25.9.RELEASE".
	 * 
	 * @param featureName the product name (required)
	 * @param simpleVersionSequence a version number such as ("1.0.4.M101"; null is acceptable)
	 * @param sourceId the source control ID, if available (null is acceptable)
	 * @return a {@link FeatureUse} assuming the item could be parsed (never null)
	 */
	public static FeatureUse getFeatureUse(String featureName, String simpleVersionSequence, String sourceId) {
		Assert.hasLength(featureName, "Feature name required");

		FeatureUse.Builder b = getFeatureUseBuilder(simpleVersionSequence);
		b.setName(featureName);
		
		if (sourceId != null && sourceId.length() > 0) {
			b.setSourceControlIdentifier(sourceId);
		}
		
		return b.build();
	}
	
	/**
	 * Delegates to {@link #getProduct(String, String, String)}, passing null as the sourceId.
	 * 
	 * @param productName the product name (required)
	 * @param simpleVersionSequence a version number such as ("1.0.4.M101"; required)
	 * @return a {@link Product} assuming the item could be parsed (never null)
	 */
	public static Product getProduct(String productName, String simpleVersionSequence) {
		return getProduct(productName, simpleVersionSequence, null);
	}
	
	/**
	 * Useful for products to obtain a {@link Product} without using the builder. This method can be
	 * used with any version number adopting the form <code>I.I.I.CI</code> where "I" means an
	 * integer, "C" means one or more characters to identify the release category, followed by an
	 * integer-based release qualifier. For example, "1.42.5.M1" or "5.25.9.RELEASE".
	 * 
	 * @param productName the product name (required)
	 * @param simpleVersionSequence a version number such as ("1.0.4.M101"; required)
	 * @param sourceId the source control ID, if available (null is acceptable)
	 * @return a {@link Product} assuming the item could be parsed (never null)
	 */
	public static Product getProduct(String productName, String simpleVersionSequence, String sourceId) {
		Assert.hasLength(productName, "Product name required");
		Assert.hasLength(simpleVersionSequence, "Simple version sequence required");
		
		Product.Builder b = getProductBuilder(simpleVersionSequence);
		b.setName(productName);
		
		if (sourceId != null && sourceId.length() > 0) {
			b.setSourceControlIdentifier(sourceId);
		}
		
		return b.build();
	}
	
	/**
	 * Obtains product version information from the OSGi Manifest headers. Note the presented class must be
	 * compiled into a .jar that contains a META-INF/MANFIEST.MF file.
	 * 
	 * @param clazz a class belonging to a bundle which contains the manifest information (required)
	 * @param productName the name of the product (required)
	 * @return the product (never null)
	 */
	public static Product getProductFromManifest(Class<?> clazz, String productName) {
		try {
			String classContainer = clazz.getProtectionDomain().getCodeSource().getLocation().toString();
			if (classContainer.endsWith(".jar")) {
				URL manifestUrl = new URL("jar:" + classContainer + "!/META-INF/MANIFEST.MF");
				Manifest manifest = new Manifest(manifestUrl.openStream());
				return getProductFromManifest(manifest, productName);
			}
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}
		
		throw new IllegalStateException("Unable to load manifest for class '" + clazz.getName() + "'");
	}
	
	/**
	 * Obtains product version information from the OSGi Manifest "Bundle-Version" (required) and 
	 * "Git-Commit-Hash" (optional) headers.
	 * 
	 * @param manifest the manifest to parse (required)
	 * @param productName the name of the product (required)
	 * @return the product (never null)
	 */
	public static Product getProductFromManifest(Manifest manifest, String productName) {
		String bundleVersion = null;
		String gitCommitHash = null;

		bundleVersion = manifest.getMainAttributes().getValue("Bundle-Version");
		Assert.notNull(bundleVersion, "Manifest does not contain a 'Bundle-Version' value");
		gitCommitHash = manifest.getMainAttributes().getValue("Git-Commit-Hash");

		return getProduct(productName, bundleVersion, gitCommitHash);
	}
	
	/**
	 * Processes an input String, returning a list of identified numbers. Treats non-numeric characters as the
	 * end of a given number.
	 * 
	 * <p>
	 * This continues until the passed maximum desired value is reached. No more than this many numbers will be
	 * returned. The character immediately following the final number is ignored, as it is assumed to be a
	 * separator (which is the most common case).
	 * 
	 * <p>
	 * Examples:
	 * <ul>
	 * <li>input ("1.6.0_20-b02", Integer.MAX_VALUE) will yield a [1, 6, 0, 20, 2] and "" as the remainder</li>
	 * <li>input ("1.6.0_20-b02", 3) will yield a [1, 6, 0] and "20-b02" as the remainder</li>
	 * <li>input ("2.6.31-22-generic", Integer.MAX_VALUE) will yield [2, 6, 31, 22] and "" as the remainder</li>
	 * <li>input ("2.6.31-22-generic", 3) will yield [2, 6, 31] and "22-generic" as the remainder</li>
	 * <li>input ("1.2.3.RELEASE", Integer.MAX_VALUE) will yield [1, 2, 3] and "RELEASE" as the remainder</li>
	 * <li>input ("1.2.3.RELEASE", 2) will yield [1, 2] and "3.RELEASE" as the remainder</li>
	 * 
	 * @param input a string to parse (required)
	 * @param maximumDesired how many numbers to return as a maximum (use {@link Integer#MAX_VALUE} if all needed)
	 * @return the located numbers, plus the remainder of the string (never null, but may be empty)
	 */
	public static GetNumbersResult getNumbers(String input, int maximumDesired) {
		GetNumbersResult result = new GetNumbersResult();
		result.numbers = new ArrayList<Integer>();
		StringBuilder currentNumber = new StringBuilder();
		
		char[] characters = input.toCharArray();
		for (int i = 0; i < characters.length; i++) {
			if (Character.isDigit(characters[i])) {
				currentNumber.append(characters[i]);
			} else {
				// Capture the number (if any) which we have found so far
				if (currentNumber.length() > 0) {
					try {
						addIfPossible(currentNumber, result.numbers);
						currentNumber = new StringBuilder();
					} catch (NumberFormatException shouldNotHappen) {}
				}
				// Determine if we're ready to abort
				if (result.numbers.size() == maximumDesired) {
					// Ignoring the current (presumed) separator character, get the rest of the input string
					if (characters.length > i+1) {
						StringBuilder rest = new StringBuilder();
						for (int x = i+1; x < characters.length; x++) {
							rest.append(characters[x]);
						}
						result.restOfString = rest.toString();
						return result;
					}
				}
			}
		}
		
		if (currentNumber.length() > 0) {
			addIfPossible(currentNumber, result.numbers);
			// No need to handle rest of string, as we ended by capturing a numeric digit going into the number list
		}
		
		return result;
	}
	
	private static void addIfPossible(StringBuilder input, List<Integer> list) {
		try {
			Integer i = new Integer(input.toString());
			list.add(i);
		} catch (NumberFormatException shouldNotHappen) {
			// Probably an integer overflow
			list.add(Integer.MAX_VALUE);
		}
	}

	// package protected for unit test use (thus we don't care about the list's mutability, as it's only used within our package)
	static class GetNumbersResult {
		List<Integer> numbers;
		String restOfString = "";
	}
}
