/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package agent.frida.model.impl;

import java.nio.ByteBuffer;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;

import agent.frida.frida.FridaRegionInfo;
import agent.frida.manager.*;
import agent.frida.manager.cmd.FridaReadKernelMemoryCommand;
import agent.frida.manager.cmd.FridaWriteKernelMemoryCommand;
import agent.frida.manager.impl.FridaManagerImpl;
import agent.frida.model.iface2.FridaModelTargetMemoryContainer;
import agent.frida.model.iface2.FridaModelTargetMemoryRegion;
import agent.frida.model.methods.*;
import ghidra.dbg.error.DebuggerMemoryAccessException;
import ghidra.dbg.error.DebuggerModelAccessException;
import ghidra.dbg.target.TargetObject;
import ghidra.dbg.target.schema.*;
import ghidra.dbg.target.schema.TargetObjectSchema.ResyncMode;
import ghidra.program.model.address.Address;
import ghidra.util.Msg;
import ghidra.util.datastruct.WeakValueHashMap;

@TargetObjectSchemaInfo(
	name = "Memory",
	elementResync = ResyncMode.ALWAYS,
	elements = {
		@TargetElementType(type = FridaModelTargetMemoryRegionImpl.class)
	},
	attributes = {
		@TargetAttributeType(type = Object.class)
	},
	canonicalContainer = true)
public class FridaModelTargetKernelMemoryContainerImpl extends FridaModelTargetObjectImpl
		implements FridaModelTargetMemoryContainer {

	protected final FridaModelTargetKernelImpl kernel;
	protected final FridaModelTargetMemoryScanImpl scan;
	protected final FridaModelTargetMemoryProtectImpl prot;
	protected final FridaModelTargetMemoryWatchImpl watch;
	protected final  FridaModelTargetUnloadScriptImpl unload;

	protected final Map<String, FridaModelTargetMemoryRegionImpl> memoryRegions =
		new WeakValueHashMap<>();


	public FridaModelTargetKernelMemoryContainerImpl(FridaModelTargetKernelImpl kernel) {
		super(kernel.getModel(), kernel, "Memory", "MemoryContainer");
		this.kernel = kernel;
		
		this.scan = new FridaModelTargetMemoryScanImpl(this, true);
		this.prot = new FridaModelTargetMemoryProtectImpl(this, true);
		this.watch = new FridaModelTargetMemoryWatchImpl(this);
		this.unload = new FridaModelTargetUnloadScriptImpl(this, watch.getName());
		this.changeAttributes(List.of(), List.of(), Map.of( //
			scan.getName(), scan, //
			prot.getName(), prot, //
			watch.getName(), watch, //
			unload.getName(), unload //
		), "Initialized");
		
		getManager().addEventsListener(this);
		requestElements(false);
	}

	@Override
	public CompletableFuture<Void> requestElements(boolean refresh) {
		if (refresh) {
			listeners.fire.invalidateCacheRequested(this);
		}
		return getManager().listKernelMemory();
	}

	@Override
	public synchronized FridaModelTargetMemoryRegion getTargetMemory(FridaMemoryRegionInfo region) {
		TargetObject targetObject = getMapObject(region);
		if (targetObject != null) {
			FridaModelTargetMemoryRegion targetRegion = (FridaModelTargetMemoryRegion) targetObject;
			targetRegion.setModelObject(region);
			return targetRegion;
		}
		return new FridaModelTargetMemoryRegionImpl(this, region);
	}

	private byte[] readAssist(Address address, ByteBuffer buf, long offset, RangeSet<Long> set) {
		if (set == null) {
			return new byte[0];
		}
		Range<Long> range = set.rangeContaining(offset);
		if (range == null) {
			throw new DebuggerMemoryAccessException("Cannot read at " + address);
		}
		listeners.fire.memoryUpdated(getProxy(), address, buf.array());
		return Arrays.copyOf(buf.array(), (int) (range.upperEndpoint() - range.lowerEndpoint()));
	}

	private void writeAssist(Address address, byte[] data) {
		listeners.fire.memoryUpdated(getProxy(), address, data);
	}

	@Override
	public CompletableFuture<byte[]> readMemory(Address address, int length) {
		return model.gateFuture(doReadMemory(address, length));
	}

	protected CompletableFuture<byte[]> doReadMemory(Address address, int length) {
		FridaManagerImpl manager = getManager();
		if (manager.isWaiting()) {
			throw new DebuggerModelAccessException(
				"Cannot process command readMemory while engine is waiting for events");
		}
		ByteBuffer buf = ByteBuffer.allocate(length);
		long offset = address.getOffset();
		if (!manager.isKernelMode() || address.getAddressSpace().getName().equals("ram")) {
			return manager
					.execute(new FridaReadKernelMemoryCommand(manager, address, buf, buf.remaining()))
					.thenApply(set -> {
						return readAssist(address, buf, offset, set);
					});
		}
		return CompletableFuture.completedFuture(new byte[length]);
	}

	@Override
	public CompletableFuture<Void> writeMemory(Address address, byte[] data) {
		return model.gateFuture(doWriteMemory(address, data));
	}

	protected CompletableFuture<Void> doWriteMemory(Address address, byte[] data) {
		FridaManagerImpl manager = getManager();
		if (manager.isWaiting()) {
			throw new DebuggerModelAccessException(
				"Cannot process command writeMemory while engine is waiting for events");
		}
		ByteBuffer buf = ByteBuffer.wrap(data);
		if (!manager.isKernelMode() || address.getAddressSpace().getName().equals("ram")) {
			return manager
					.execute(new FridaWriteKernelMemoryCommand(manager, address, buf, buf.remaining()))
					.thenAccept(___ -> {
						writeAssist(address, data);
					});
		}
		return CompletableFuture.completedFuture(null);
	}

	@Override
	public void regionAdded(FridaProcess process, FridaRegionInfo info, int index, FridaCause cause) {
		FridaModelTargetMemoryRegion targetRegion;
		FridaMemoryRegionInfo region = info.getRegion(index);
		synchronized (this) {
			/**
			 * It's not a good idea to remove "stale" entries. If the entry's already present, it's
			 * probably because several modules were loaded at once, at it has already had its
			 * sections loaded. Removing it will cause it to load all module sections again!
			 */
			//modulesByName.remove(name);
			targetRegion = getTargetMemory(region);
		}
		if (targetRegion == null) {
			Msg.error(this, "Region " + region.getRangeAddress() + " not found!");
			return;
		}
		changeElements(List.of(), List.of(targetRegion), Map.of(), "Added");
	}

	@Override
	public void regionReplaced(FridaProcess process, FridaRegionInfo info, int index, FridaCause cause) {
		FridaMemoryRegionInfo region = info.getRegion(index);
		changeElements(List.of(), List.of(getTargetMemory(region)), Map.of(), "Replaced");
		FridaModelTargetMemoryRegion targetRegion = getTargetMemory(region);
		changeElements(List.of(), List.of(targetRegion), Map.of(), "Replaced");
	}

	@Override
	public void regionRemoved(FridaProcess process, FridaRegionInfo info, int index, FridaCause cause) {
		FridaModelTargetMemoryRegion targetRegion = getTargetMemory(info.getRegion(index));
		if (targetRegion != null) {
			FridaModelImpl impl = (FridaModelImpl) model;
			impl.deleteModelObject(targetRegion.getModelObject());
		}
		changeElements(List.of(targetRegion.getName()), List.of(), Map.of(), "Removed");
	}
}
