/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.cmd.formats;

import ghidra.app.plugin.core.analysis.AnalysisWorker;
import ghidra.app.plugin.core.analysis.AutoAnalysisManager;
import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.MemoryByteProvider;
import ghidra.app.util.bin.format.elf.ElfConstants;
import ghidra.app.util.bin.format.elf.ElfDynamic;
import ghidra.app.util.bin.format.elf.ElfDynamicTable;
import ghidra.app.util.bin.format.elf.ElfDynamicType;
import ghidra.app.util.bin.format.elf.ElfException;
import ghidra.app.util.bin.format.elf.ElfHeader;
import ghidra.app.util.bin.format.elf.ElfProgramHeader;
import ghidra.app.util.bin.format.elf.ElfRelocationTable;
import ghidra.app.util.bin.format.elf.ElfSectionHeader;
import ghidra.app.util.bin.format.elf.ElfStringTable;
import ghidra.app.util.bin.format.elf.ElfSymbol;
import ghidra.app.util.bin.format.elf.ElfSymbolTable;
import ghidra.app.util.importer.MessageLog;
import ghidra.framework.cmd.BinaryAnalysisCommand;
import ghidra.framework.options.Options;
import ghidra.program.flatapi.FlatProgramAPI;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.data.ArrayDataType;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.Structure;
import ghidra.program.model.listing.CodeUnit;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.symbol.RefType;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.util.CodeUnitInsertionException;
import ghidra.util.Msg;
import ghidra.util.StringUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.task.TaskMonitor;
import java.util.Arrays;
import org.apache.commons.lang3.StringUtils;

public class ElfBinaryAnalysisCommand
extends FlatProgramAPI
implements BinaryAnalysisCommand,
AnalysisWorker {
    private MessageLog messages = new MessageLog();

    @Override
    public boolean canApply(Program program) {
        try {
            Options options = program.getOptions("Program Information");
            String format = options.getString("Executable Format", null);
            if (!"Raw Binary".equals(format)) {
                return false;
            }
            Memory memory = program.getMemory();
            byte[] magicBytes = new byte[ElfConstants.MAGIC_BYTES.length];
            memory.getBytes(program.getAddressFactory().getDefaultAddressSpace().getAddress(0L), magicBytes);
            return Arrays.equals(magicBytes, ElfConstants.MAGIC_BYTES);
        }
        catch (Exception e) {
            return false;
        }
    }

    @Override
    public boolean analysisWorkerCallback(Program program, Object workerContext, TaskMonitor analysisMonitor) throws Exception, CancelledException {
        this.set(program, analysisMonitor);
        Listing listing = this.currentProgram.getListing();
        SymbolTable symbolTable = this.currentProgram.getSymbolTable();
        MemoryByteProvider provider = MemoryByteProvider.createDefaultAddressSpaceByteProvider(program, false);
        try {
            ElfHeader elf = new ElfHeader(provider, msg -> this.messages.appendMsg(msg));
            elf.parse();
            this.processElfHeader(elf, listing);
            this.processProgramHeaders(elf, listing);
            this.processSectionHeaders(elf, listing);
            this.processInterpretor(elf, provider, program);
            this.processDynamic(elf, provider, program);
            this.processSymbolTables(elf, listing, symbolTable);
            this.processStrings(elf);
            this.processRelocationTables(elf, listing);
            return true;
        }
        catch (ElfException e) {
            this.messages.appendMsg("Not a binary ELF program: ELF header not found.");
            return false;
        }
    }

    @Override
    public String getWorkerName() {
        return this.getName();
    }

    @Override
    public boolean applyTo(Program program, TaskMonitor analysisMonitor) throws Exception {
        this.set(program, analysisMonitor);
        AutoAnalysisManager aam = AutoAnalysisManager.getAnalysisManager(this.currentProgram);
        return aam.scheduleWorker(this, null, false, analysisMonitor);
    }

    @Override
    public String getName() {
        return "ELF Header Annotation";
    }

    @Override
    public MessageLog getMessages() {
        return this.messages;
    }

    private void processElfHeader(ElfHeader elf, Listing listing) throws DuplicateNameException, CodeUnitInsertionException, Exception {
        DataType elfDT = elf.toDataType();
        Address elfStart = this.addr(0L);
        this.createData(elfStart, elfDT);
        this.createFragment(elfDT.getName(), elfStart, elfDT.getLength());
    }

    private void processStrings(ElfHeader elf) throws CancelledException {
        ElfSectionHeader[] stringSections;
        Memory memory = this.currentProgram.getMemory();
        for (ElfSectionHeader stringSection : stringSections = elf.getSections(3)) {
            this.monitor.checkCanceled();
            try {
                Address addr = this.addr(stringSection.getOffset());
                Address maxAddr = addr.addNoWrap(stringSection.getSize() - 1L);
                MemoryBlock block = memory.getBlock(addr);
                if (block == null) {
                    this.messages.appendMsg("Unable to markup string table at " + addr + " - block not found");
                    continue;
                }
                if (maxAddr.compareTo((Object)block.getEnd()) > 0) {
                    this.messages.appendMsg("Truncated string table markup at " + addr + " - block too short");
                    maxAddr = block.getEnd();
                }
                addr = addr.addNoWrap(1L);
                while (addr.compareTo((Object)maxAddr) < 0) {
                    Data d = this.createAsciiString(addr);
                    addr = addr.addNoWrap((long)d.getLength());
                }
            }
            catch (AddressOverflowException | CodeUnitInsertionException e) {
                this.messages.appendMsg("Failed to markup string table: " + e.getMessage());
            }
            catch (Exception e) {
                this.messages.appendException((Throwable)e);
            }
        }
    }

    private void processSectionHeaders(ElfHeader elf, Listing listing) throws Exception {
        ElfSectionHeader[] sections = elf.getSections();
        for (int i = 0; i < sections.length; ++i) {
            this.monitor.checkCanceled();
            String name = sections[i].getNameAsString();
            DataType sectionDT = sections[i].toDataType();
            long offset = elf.e_shoff() + (long)(i * elf.e_shentsize());
            Address sectionStart = this.addr(offset);
            this.createData(sectionStart, sectionDT);
            this.createFragment(sectionDT.getName(), sectionStart, sectionDT.getLength());
            CodeUnit cu = listing.getCodeUnitAt(this.addr(offset));
            cu.setComment(3, "#" + i + ") " + name + " at 0x" + Long.toHexString(sections[i].getAddress()));
            if (sections[i].getType() == 8 || sections[i].getSize() == 0L || sections[i].isInvalidOffset()) continue;
            Address dataStart = this.addr(sections[i].getOffset());
            this.createFragment(name + "_DATA", dataStart, sections[i].getSize());
            try {
                this.createLabel(dataStart, name, true, SourceType.ANALYSIS);
            }
            catch (Exception exception) {
                // empty catch block
            }
            cu = listing.getCodeUnitAt(dataStart);
            cu.setComment(1, sections[i].getNameAsString() + " Size: 0x" + Long.toHexString(sections[i].getSize()));
        }
    }

    private void processProgramHeaders(ElfHeader elf, Listing listing) throws Exception {
        int headerCount = elf.getProgramHeaderCount();
        int size = elf.e_phentsize() * headerCount;
        if (size == 0) {
            return;
        }
        Structure phStructDt = (Structure)elf.getProgramHeaders()[0].toDataType();
        phStructDt = phStructDt.clone(listing.getDataTypeManager());
        ArrayDataType arrayDt = new ArrayDataType((DataType)phStructDt, headerCount, size);
        Data array = this.createData(this.addr(elf.e_phoff()), (DataType)arrayDt);
        this.createFragment(phStructDt.getName(), array.getMinAddress(), array.getLength());
        ElfProgramHeader[] programHeaders = elf.getProgramHeaders();
        for (int i = 0; i < programHeaders.length; ++i) {
            this.monitor.checkCanceled();
            Data d = array.getComponent(i);
            d.setComment(0, programHeaders[i].getComment());
            Address addr = this.addr(programHeaders[i].getOffset());
            this.createLabel(addr, programHeaders[i].getTypeAsString(), true, SourceType.ANALYSIS);
        }
    }

    private void processInterpretor(ElfHeader elf, ByteProvider provider, Program program) throws CancelledException {
        for (ElfProgramHeader programHeader : elf.getProgramHeaders(3)) {
            this.monitor.checkCanceled();
            long offset = programHeader.getOffset();
            if (offset == 0L) {
                Msg.warn((Object)this, (Object)" Dynamic table appears to have been stripped from binary");
                return;
            }
            try {
                this.createAsciiString(this.addr(offset));
            }
            catch (AddressOverflowException | CodeUnitInsertionException e) {
                this.messages.appendMsg("Failed to markup PT_INTERP string: " + e.getMessage());
            }
            catch (Exception e) {
                this.messages.appendException((Throwable)e);
            }
        }
    }

    private void processDynamic(ElfHeader elf, ByteProvider provider, Program program) throws CancelledException {
        ElfDynamicTable dynamicTable = elf.getDynamicTable();
        if (dynamicTable == null) {
            return;
        }
        try {
            Address addr = this.addr(dynamicTable.getFileOffset());
            program.getSymbolTable().createLabel(addr, "_DYNAMIC", SourceType.ANALYSIS);
            ElfDynamic[] dynamics = dynamicTable.getDynamics();
            DataType structArray = dynamicTable.toDataType();
            Data dynamicTableData = this.createData(addr, structArray);
            BinaryReader reader = new BinaryReader(provider, !program.getMemory().isBigEndian());
            for (int i = 0; i < dynamics.length; ++i) {
                this.monitor.checkCanceled();
                Data dynamicData = dynamicTableData.getComponent(i);
                if (dynamicData == null) {
                    return;
                }
                int tagType = dynamics[i].getTag();
                ElfDynamicType dynamicType = elf.getDynamicType(tagType);
                String comment = dynamicType != null ? dynamicType.name + " - " + dynamicType.description : "DT_0x" + StringUtilities.pad((String)Integer.toHexString(tagType), (char)'0', (int)8);
                dynamicData.setComment(0, comment);
                Data valueData = dynamicData.getComponent(1);
                if (dynamicType != null) {
                    if (dynamicType.valueType == ElfDynamicType.ElfDynamicValueType.ADDRESS) {
                        this.addDynamicReference(elf, dynamics[i], valueData.getAddress(), program);
                        continue;
                    }
                    if (dynamicType.valueType != ElfDynamicType.ElfDynamicValueType.STRING) continue;
                    this.addDynamicStringComment(elf, dynamics[i], valueData, reader, program);
                    continue;
                }
                this.addDynamicReference(elf, dynamics[i], valueData.getAddress(), program);
            }
        }
        catch (CancelledException e) {
            throw e;
        }
        catch (Exception e) {
            this.messages.appendMsg("Could not markup dynamic section: " + e);
        }
    }

    private void addDynamicStringComment(ElfHeader elf, ElfDynamic dynamic, Data data, BinaryReader reader, Program program) {
        String str;
        ElfStringTable dynamicStringTable = elf.getDynamicStringTable();
        if (dynamicStringTable != null && (str = dynamicStringTable.readString(reader, dynamic.getValue())) != null && str.length() != 0) {
            data.setComment(0, str);
        }
    }

    private void addDynamicReference(ElfHeader elf, ElfDynamic dynamic, Address fromAddr, Program program) {
        long dynamicRefAddr = dynamic.getValue();
        ElfProgramHeader programLoadHeader = elf.getProgramLoadHeaderContaining(dynamicRefAddr);
        if (programLoadHeader == null) {
            return;
        }
        Address refAddr = this.addr(programLoadHeader.getOffset(dynamicRefAddr));
        program.getReferenceManager().addMemoryReference(fromAddr, refAddr, RefType.DATA, SourceType.ANALYSIS, 0);
        try {
            this.createLabel(refAddr, "_" + dynamic.getTagAsString(), true, SourceType.ANALYSIS);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void processSymbolTables(ElfHeader elf, Listing listing, SymbolTable symbolTable) throws CancelledException {
        ElfSymbolTable[] symbolTables;
        this.monitor.setMessage("Processing symbol tables...");
        for (ElfSymbolTable symbolTable2 : symbolTables = elf.getSymbolTables()) {
            this.monitor.checkCanceled();
            Address symbolTableAddr = this.addr(symbolTable2.getFileOffset());
            try {
                DataType symbolTableDT = symbolTable2.toDataType();
                this.createData(symbolTableAddr, symbolTableDT);
            }
            catch (Exception e) {
                this.messages.appendMsg("Could not markup symbol table: " + e);
                return;
            }
            ElfSymbol[] symbols = symbolTable2.getSymbols();
            for (int j = 0; j < symbols.length; ++j) {
                if (this.monitor.isCancelled()) {
                    return;
                }
                String name = symbols[j].getNameAsString();
                if (StringUtils.isBlank((CharSequence)name)) continue;
                try {
                    Address currAddr = symbolTableAddr.add((long)(j * symbolTable2.getEntrySize()));
                    listing.setComment(currAddr, 0, name + " at 0x" + Long.toHexString(symbols[j].getValue()));
                    continue;
                }
                catch (Exception e) {
                    this.messages.appendMsg("Could not markup symbol table: " + e);
                }
            }
        }
    }

    private Address addr(long offset) {
        return this.currentProgram.getAddressFactory().getDefaultAddressSpace().getAddress(offset);
    }

    private void processRelocationTables(ElfHeader elf, Listing listing) throws CancelledException {
        ElfRelocationTable[] relocationTables;
        this.monitor.setMessage("Processing relocation tables...");
        for (ElfRelocationTable relocationTable : relocationTables = elf.getRelocationTables()) {
            this.monitor.checkCanceled();
            ElfSectionHeader relocationSection = relocationTable.getTableSectionHeader();
            String relocSectionName = "<section-not-found>";
            if (relocationSection != null) {
                relocSectionName = relocationSection.getNameAsString();
            }
            Address relocationTableAddress = this.addr(relocationTable.getFileOffset());
            try {
                DataType dataType = relocationTable.toDataType();
                if (dataType != null) {
                    this.createData(relocationTableAddress, dataType);
                    continue;
                }
                listing.setComment(relocationTableAddress, 1, "ELF Relocation Table (markup not yet supported)");
            }
            catch (Exception e) {
                this.messages.appendMsg("Could not markup relocation table for " + relocSectionName + " - " + e);
            }
        }
    }
}

