// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Text.StructuredFormat

    // Breakable block layout implementation.
    // This is a fresh implementation of pre-existing ideas.

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open Microsoft.FSharp.Primitives.Basics.List
      
    /// A joint, between 2 layouts, is either:
    ///  - unbreakable, or
    ///  - breakable, and if broken the second block has a given indentation.
    type Joint =
       Unbreakable
     | Breakable of int
     | Broken of int

    /// Leaf juxt,data,juxt
    /// Node juxt,left,juxt,right,juxt and joint
    ///
    /// If either juxt flag is true, then no space between words.
    type Layout =
       Leaf of bool * obj * bool
     | Node of bool * layout * bool * layout * bool * joint
     | Attr of string * (string * string) list * layout

    and layout = Layout
    and joint = Joint

    type IEnvironment = 
      interface
        abstract GetLayout : obj -> layout
        abstract MaxColumns : int
        abstract MaxRows : int
      end
      
    type IFormattable = 
      interface
        abstract GetLayout : IEnvironment -> layout
      end


    module LayoutOps = 
        let rec juxtLeft = function
            Leaf (jl,text,jr)         -> jl
          | Node (jl,l,jm,r,jr,joint) -> jl
          | Attr (tag,attrs,l)        -> juxtLeft l

        let rec juxtRight = function
            Leaf (jl,text,jr)         -> jr
          | Node (jl,l,jm,r,jr,joint) -> jr
          | Attr (tag,attrs,l)        -> juxtRight l

        let mkNode l r joint =
           let jl = juxtLeft  l in
           let jm = juxtRight l || juxtLeft r in
           let jr = juxtRight r in
           Node(jl,l,jm,r,jr,joint)


        // constructors

        let objL   obj = Leaf (false,obj,false)
        let sLeaf  (l,(str:string),r) = Leaf (l,(str:>obj),r)
        let wordL  str = sLeaf (false,str,false)
        let sepL   str = sLeaf (true ,str,true)   
        let rightL str = sLeaf (true ,str,false)   
        let leftL  str = sLeaf (false,str,true)
        let emptyL = sLeaf (true,"",true)
        let isEmptyL = function 
         | Leaf(true,s,true) -> 
            match s with 
            | :? string as s -> s = "" 
            | _ -> false
         | _ -> false
         
        let aboveL  l r = mkNode l r (Broken 0)
        let joinN i l r = mkNode l r (Breakable i)                                      
        let join  = joinN 0
        let join1 = joinN 1
        let join2 = joinN 2
        let join3 = joinN 3

        let tagAttrL tag attrs l = Attr(tag,attrs,l)
        let tagL tag l = tagAttrL tag [] l 


        // constructors derived

        let apply2 f l r = if isEmptyL l then r else
                           if isEmptyL r then l else f l r

        let ($$)  l r  = mkNode l r (Unbreakable)
        let (++)  l r  = mkNode l r (Breakable 0)
        let (--)  l r  = mkNode l r (Breakable 1)
        let (---) l r  = mkNode l r (Breakable 2)
        let (@@)   l r = apply2 (fun l r -> mkNode l r (Broken 0)) l r
        let (@@-)  l r = apply2 (fun l r -> mkNode l r (Broken 1)) l r
        let (@@--) l r = apply2 (fun l r -> mkNode l r (Broken 2)) l r

        let tagListL tagger = function
            []    -> emptyL
          | [x]   -> x
          | x::xs ->
              let rec process' prefixL = function
                  []    -> prefixL
                | y::ys -> process' ((tagger prefixL) ++ y) ys
              in  process' x xs
            
        let commaListL x = tagListL (fun prefixL -> prefixL $$ rightL ",") x
        let semiListL x  = tagListL (fun prefixL -> prefixL $$ rightL ";") x
        let spaceListL x = tagListL (fun prefixL -> prefixL) x
        let sepListL x y = tagListL (fun prefixL -> prefixL $$ x) y

        let bracketL l = leftL "(" $$ l $$ rightL ")"
        let tupleL xs = bracketL (sepListL (sepL ",") xs)
        let aboveListL = function
            []    -> emptyL
          | [x]   -> x
          | x::ys -> List.fold_left (fun pre y -> pre @@ y) x ys

        let optionL xL = function
            None   -> wordL "None"
          | Some x -> wordL "Some" -- (xL x)

        let listL xL xs = leftL "[" $$ sepListL (sepL ";") (map xL xs) $$ rightL "]"

        let squareBracketL x = leftL "[" $$ x $$ rightL "]"    
        let braceL         x = leftL "{" $$ x $$ rightL "}"

        let unfoldL (itemL   : 'a -> layout)
                    (project : 'z -> ('a * 'z) option)
                    (z : 'z)
                    maxLength =
          let rec consume n z =
            match project z with
                None       -> []  (* exhaused input *)
              | Some (x,z) -> if n<=0 then [wordL "..."]               (* hit print_length limit *)
                                      else itemL x :: consume (n-1) z  (* cons recursive... *)
          consume maxLength z  

