(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.FSharp.Compiler.Formats
open Microsoft.FSharp.Compiler 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
F#*)

open Ast
open Tast
open Tastops
open Env
open Csolve

(* format strings are unicode *)
let get fmt i = Char.chr (Bytes.get fmt i), Char.chr (Bytes.get fmt (i+1))

type format_item = Simple of typ | FuncAndVal 

let copy_and_fixup_format_typar amap m tp = 
    let _,_,tinst = freshen_and_fixup_typars amap m TyparFlexible [] [] [tp] in
    List.hd tinst

let lowestDefaultPriority = 0 (* See comment on TTyparDefaultsToType *)

let flexible_format_typar amap g m tys dflt = 
    let tp = new_typar (TyparRigid,Typar(mksyn_id m "fmt",CompleteStaticReq,true),false,[]) in
    fixup_typar_constraints tp [ TTyparSimpleChoice (tys,m); TTyparDefaultsToType (lowestDefaultPriority,dflt,m)];
    copy_and_fixup_format_typar amap m tp

let flexible_int_format_typar amap g m = 
    flexible_format_typar amap g m [ g.byte_ty; g.int16_ty; g.int32_ty; g.int64_ty;  g.sbyte_ty; g.uint16_ty; g.uint32_ty; g.uint64_ty;g.nativeint_ty;g.unativeint_ty; ] g.int_ty
    
    
let flexible_float_format_typar amap g m = 
    flexible_format_typar amap g m [ g.float_ty; g.float32_ty; ] g.float_ty

let is_digit c = ('0' <= c && c <= '9')

let parse_format amap m g fmt bty cty dty = 
  let len = Bytes.length fmt in 
  let rec go acc i = 
   if i+1 >= len then 
        let argtys = List.rev acc in 
        let aty = List.fold_right (-->) argtys dty in
        let ety = mk_tupled_ty g argtys in
        aty,ety 
   else
    match get fmt i with
    | '%','\000' ->
        let i = i+2 in 
        if i+1 >= len then failwith "missing format specifier";
        let rec flags i = 
          match get fmt i with
          | ('-' | '+' | '0' | ' '),'\000' ->  flags(i+2)
          | '#','\000' -> failwith "The # formatting modifier is invalid in F#."; 
          | _ -> i in

        let rec digits_precision i = 
          if i+1 >= len then failwith "bad precision in format specifier";
          match get fmt i with
          | c,'\000' when is_digit c -> digits_precision (i+2)
          | _ -> i in 

        let precision i = 
          if i+1 >= len then failwith "bad width in format specifier";
          match get fmt i with
          | c,'\000' when is_digit c -> false,digits_precision (i+2)
          | '*','\000' -> true,(i+2)
          | _ -> false,i in

        let optional_dot_and_precision i = 
          if i+1 >= len then failwith "bad width in format specifier";
          match get fmt i with
          | '.','\000' -> precision (i+2)
          | _ -> false,i in

        let rec digits_width_and_precision i = 
          if i+1 >= len then failwith "bad width in format specifier";
          match get fmt i with
          | c,'\000' when is_digit c -> digits_width_and_precision (i+2)
          | _ -> optional_dot_and_precision i in

        let width_and_precision i = 
          if i+1 >= len then failwith "bad width in format specifier";
          match get fmt i with
          | c,'\000' when is_digit c -> false,digits_width_and_precision i
          | '*','\000' -> true,optional_dot_and_precision (i+2)
          | _ -> false,optional_dot_and_precision i in

        let i = flags i in 

        let widthArg,(precisionArg,i) = width_and_precision i in 

        if i+1 >= len then failwith "bad precision in format specifier";

        let acc = if precisionArg then g.int_ty :: acc else acc in 

        let acc= if widthArg then g.int_ty :: acc else acc in 

        begin match get fmt i with
        | '%','\000' -> go acc (i+2) 

        | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'),'\000' -> 
            go (flexible_int_format_typar amap g m :: acc) (i+2)

        | ('h' | 'H' | 'l' | 'L' as  c),'\000' ->
            let i = i+2 in 
            if i+1 >= len then failwith "bad format specifier: %l should be %ld, %lx etc.";
            begin match c with 
            | 'L' | 'l' -> warning(OCamlCompatibility("The 'l' or 'L' in this format specifier is unnecessary except in code cross-compiled with OCaml. In F# code you can use %d, %x, %o or %u instead, which are overloaded to work with all basic integer types",m));
            | _ -> warning(Error("The 'h' or 'H' in this format specifier is unnecessary exceptYou can use %d, %x, %o or %u instead, which are overloaded to work with all basic integer types",m));
            end;
            begin match get fmt i with
            | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'), '\000' -> 
                go (flexible_int_format_typar amap g m :: acc)  (i+2)
            | _ -> failwith "bad format specifier (after n)"
            end

        | 'M','\000' -> 
            go (g.decimal_ty :: acc) (i+2)

        | 'n','\000' ->
            let i = i+2 in 
            if i+1 >= len then failwith "bad format specifier: %n should be %nd, %nx etc.";
            warning(Error("%n format patterns are deprecated. You can use %d instead, which is overloaded to work with all basic integer types",m));
            begin match get fmt i with
            | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'), '\000' -> 
                go (g.nativeint_ty  :: acc) (i+2)
            | _ -> failwith "bad format specifier (after l)"
            end

        | 'U','\000' -> 
            let i = i+2 in 
            if i+1 >= len then failwith "bad format specifier: %U should be %Uld, %Ulx etc.";
            warning(Error("%U format patterns are deprecated. You can use %u instead, which is overloaded to work with all basic integer types",m));
            begin match get fmt i with
            | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'), '\000' -> 
                go (g.uint32_ty  :: acc) (i+2)

            | 'l','\000' ->
                let i = i+2 in 
                if i+1 >= len then failwith "bad format specifier: %Ul should be %Uld, %Ulx etc.";
                begin match get fmt i with
                | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'), '\000' -> 
                    go (g.uint32_ty  :: acc) (i+2)
                | _ -> failwith "bad format specifier (after Ul)"
                end

            | 'n','\000' ->
                let i = i+2 in 
                if i+1 >= len then failwith "bad format specifier: %Un should be %Uld, %Ulx etc.";
                begin match get fmt i with
                | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'), '\000' ->  
                    go (g.unativeint_ty  :: acc) (i+2)
                | _ -> failwith "bad format specifier (after Un)"
                end

            | 'L','\000' ->
                let i = i+2 in 
                if i+1 >= len then failwith "bad format specifier: %UL should be %Uld, %Ulx etc.";
                begin match get fmt i with
                | ('d' | 'i' | 'o' | 'u' | 'x' | 'X'), '\000' -> 
                    go (g.uint64_ty  :: acc) (i+2)
                | _ -> failwith "bad format specifier (after UL)"
                end
            | _ -> failwith "bad format specifier (after U)"
            end

        | ('f' | 'F' | 'e' | 'E' | 'g' | 'G'),'\000' ->  
            go ((flexible_float_format_typar amap g m) :: acc) (i+2)

        | 'b','\000' -> 
            go (g.bool_ty  :: acc) (i+2)

        | 'c','\000' -> 
            go (g.char_ty  :: acc) (i+2)

        | 's','\000' -> 
            go (g.string_ty  :: acc) (i+2)

        | 'O','\000' -> 
            go (new_inference_typ () :: acc)  (i+2)

        | 'A','\000' -> 
            go (new_inference_typ () :: acc)  (i+2)

        | 'a','\000' ->
            let xty = new_inference_typ () in 
            let fty = bty --> (xty --> cty) in
            go (xty ::  fty :: acc) (i+2)

        | 't','\000' -> 
            go ((bty --> cty) :: acc)  (i+2)

        | c, _ -> failwith ("bad format specifier: '%"^(String.make 1 c)^"'") 
        end
        
    | _ -> go acc (i+2) in 
  go [] 0

