(* -------------------------------------------------------------------- 
 * (c) Microsoft Corporation. All rights reserved 
 * -------------------------------------------------------------------- *)

(*F# 
module Microsoft.Research.AbstractIL.Internal.Nums 
F#*)

(*IF-OCAML*)
type u8 = U8 of int
type u16 = U16 of int 
type i8 = I8 of int
type i16 = I16 of int 
type u32 = U32 of int64
type i32 = int32 
type i64 = int64 
type u64 = U64 of int64 
type ieee32 = IEEE32 of int32
type ieee64 = IEEE64 of int64
type unichar = u16
(*ENDIF-OCAML*)

(*F#
type u8 = System.Byte
type u16  = System.UInt16
type i8  = System.SByte 
type i16  = System.Int16 
type u32 = System.UInt32 
type i32 = int32
type i64 = int64
type u64 = System.UInt64 
type ieee32 = System.Single
type ieee64 = System.Double
type unichar = System.Char
F#*)

(* Widening *)

(*IF-OCAML*)
let u8_to_int (U8 x) = x
let i8_to_int (I8 x) = x
let u16_to_int (U16 x) = x
let i16_to_int (I16 x) = x

let i8_to_i32 (I8 x) = Int32.of_int x
let i16_to_i32 (I16 x) = Int32.of_int x
let u16_to_i32 (U16 x) = Int32.of_int x
let u8_to_i32 (U8 x) = Int32.of_int x
let int_to_i32 x = Int32.of_int x

let int_to_i64 x = Int64.of_int x
let u32_to_i64 (U32 x) = x
let i32_to_i64 x = Int64.of_int32 x

(* Narrowing *)
let i32_to_i8 x = I8 (Int32.to_int x)
let i64_to_i8 x = I8 (Int64.to_int x)
let int_to_i8 x = I8 x

let i32_to_u8 x = U8 (Int32.to_int x)
let i64_to_u8 x = U8 (Int64.to_int x)
let int_to_u8 x = if x < 0 or x >= 256 then failwith "int_to_u8" else U8 x

let int_to_i16 x = (I16 x)
let i32_to_i16 x = I16 (Int32.to_int x)
let i64_to_i16 x = I16 (Int64.to_int x)

let int_to_u16 x = if x < 0 then failwith "int_to_u16" else U16 x
let i32_to_u16 x = U16 (Int32.to_int x)
let i64_to_u16 x = int_to_u16 (Int64.to_int x)

let u32_to_int (U32 x) = Int64.to_int x
let i32_to_int x = Int32.to_int x
let i64_to_int x = Int64.to_int x
let u64_to_int (U64 x) = Int64.to_int x

let i64_to_i32 x = (Int64.to_int32 x)
let u64_to_u32 (U64 x) = U32 x

let u64_to_i64 (U64 x) = x
let i64_to_u64 (x:i64) = U64 x
let u32_to_i32 (U32 x) = Int64.to_int32 x
let i32_to_u32 (x:i32) = U32 (Int64.of_int32 x)

let u16_to_unichar (x:u16) = x
let unichar_to_u16 (x:unichar) = x

(*ENDIF-OCAML*)

(*F#

let u8_to_int  (x:u8 ) : int = System.Convert.ToInt32(x)
let i8_to_int  (x:i8 ) : int = System.Convert.ToInt32(x)
let u16_to_int (x:u16) : int = System.Convert.ToInt32(x)
let i16_to_int (x:i16) : int = System.Convert.ToInt32(x)

let i8_to_i32  (x:i8)  : i32 = System.Convert.ToInt32(x)
let i16_to_i32 (x:i16) : i32 = System.Convert.ToInt32(x)
let u16_to_i32 (x:u16) : i32 = System.Convert.ToInt32(x)
let u8_to_i32  (x:u8)  : i32 = System.Convert.ToInt32(x)
let int_to_i32 (x:int) : i32 = System.Convert.ToInt32(x)

let int_to_i64 (x:int) : i64 = System.Convert.ToInt64(x)
let u32_to_i64 (x:u32) : i64 = System.Convert.ToInt64(x)
let i32_to_i64 (x:i32) : i64 = System.Convert.ToInt64(x)

(* Narrowing *)
let i32_to_i8  (x:i32) : i8  = System.Convert.ToSByte(x)
let i64_to_i8  (x:i64) : i8  = System.Convert.ToSByte(x)
let int_to_i8  (x:int) : i8  = System.Convert.ToSByte(x)

let i32_to_u8  (x:i32) : u8  = System.Convert.ToByte(x)
let i64_to_u8  (x:i64) : u8  = System.Convert.ToByte(x)
let int_to_u8  (x:int) : u8  = System.Convert.ToByte(x)

let int_to_i16 (x:int) : i16 = System.Convert.ToInt16(x)
let i32_to_i16 (x:i32) : i16 = System.Convert.ToInt16(x)
let i64_to_i16 (x:i64) : i16 = System.Convert.ToInt16(x)

let int_to_u16 (x:int) : u16 = System.Convert.ToUInt16(x)
let i32_to_u16 (x:i32) : u16 = System.Convert.ToUInt16(x)
let i64_to_u16 (x:i64) : u16 = System.Convert.ToUInt16(x)

let u32_to_int (x:u32) : int = System.Convert.ToInt32(x)
let i32_to_int (x:i32) : int = System.Convert.ToInt32(x)
let i64_to_int (x:i64) : int = System.Convert.ToInt32(x)
let u64_to_int (x:u64) : int = System.Convert.ToInt32(x)

let i64_to_i32 (x:i64) : i32 = System.Convert.ToInt32(x)
let u64_to_u32 (x:u64) : u32 = System.Convert.ToUInt32(x)

(* sign loss - these don't raise exceptions if things don't fit, but rather just cast the bits *)
let u64_to_i64 (x:u64) : i64 = Int64.of_uint64 x
let i64_to_u64 (x:i64) : u64 = UInt64.of_int64 x
let u32_to_i32 (x:u32) : i32 = Int32.of_uint32 x
let i32_to_u32 (x:i32) : u32 = UInt32.of_int32 x

let u16_to_unichar (x:u16) = Char.chr (u16_to_int x)
let unichar_to_u16 (x:unichar) = int_to_u16 (Char.code x)
F#*)

let i8_to_u8   x = let x = i8_to_int  x in int_to_u8  (if x <  0      then x + 0x100   else x)
let u8_to_i8   x = let x = u8_to_int  x in int_to_i8  (if x >= 0x80   then x - 0x100   else x)
let i16_to_u16 x = let x = i16_to_int x in int_to_u16 (if x <  0      then x + 0x10000 else x)
let u16_to_i16 x = let x = u16_to_int x in int_to_i16 (if x >= 0x8000 then x - 0x10000 else x)

(*IF-OCAML*)

let float_to_ieee64 (x:float) : ieee64 = IEEE64 (Int64.bits_of_float x)
let float_to_ieee32 (x:float) = IEEE32 (Int32.bits_of_float x)
let ieee64_to_float (IEEE64 x) : float = Int64.float_of_bits x 
let ieee32_to_float (IEEE32 x) = Int32.float_of_bits x
let ieee64_to_bits (IEEE64 x) = x
let ieee32_to_bits (IEEE32 x) = x
let bits_to_ieee64 (x:i64) = IEEE64 x
let bits_to_ieee32 x = IEEE32 x
(*ENDIF-OCAML*)

(*F#
let float32_to_ieee32 (x:float32) = x
let ieee32_to_float32 (x:ieee32)  = x
let ieee64_to_float   (x:ieee64)  = x 
let float_to_ieee64   (x:float)   = x
let ieee32_to_float (x:ieee32) = Float32.to_float x
let ieee64_to_bits x = Int64.bits_of_float   x
let ieee32_to_bits x = Int32.bits_of_float32 x
let bits_to_ieee64 x = Int64.float_of_bits   x
let bits_to_ieee32 x = Int32.float32_of_bits x
let float_to_ieee32   (x:float) = Float32.of_float x
F#*)


let int_to_ieee32 n =  float_to_ieee32 (float_of_int n)
let int_to_ieee64 n = float_to_ieee64 (float_of_int n)
let i32_to_ieee64 x = float_to_ieee64 (Int32.to_float x)
let i64_to_ieee64 x = float_to_ieee64 (Int64.to_float x)

(* Arith *)
let i32_lt (x:i32) y = x < y
let i32_ge (x:i32) y = x >= y
let i32_max (x:i32) y = max x y
let u16_ge x y = u16_to_int x >= u16_to_int y
let u16_sub x y = int_to_u16 (u16_to_int x - u16_to_int y)

let ieee64_of_eight_bytes b1 b2 b3 b4 b5 b6 b7 b8 = 
  bits_to_ieee64 (List.fold_left (fun acc b -> Int64.add (Int64.of_int b) (Int64.mul acc (Int64.of_int 2))) (Int64.of_int 0) [b1;b2;b3;b4;b5;b6;b7;b8])

let ieee32_of_four_bytes b1 b2 b3 b4 = 
  bits_to_ieee32 (List.fold_left (fun acc b -> Int32.add (Int32.of_int b) (Int32.mul acc (Int32.of_int 2))) (Int32.of_int 0) [b1;b2;b3;b4])

let i32_zero = Int32.zero
let u16_zero = int_to_u16 0

let output_u8 os x = output_string os (string_of_int (u8_to_int x))
let output_i8 os x = output_string os (string_of_int (i8_to_int x))
let output_u16 os x = output_string os (string_of_int (u16_to_int x))
let output_i16 os x = output_string os (string_of_int (i16_to_int x))
let output_u32 os x = output_string os (Int64.to_string (u32_to_i64 x))
let output_i32 os x = output_string os (Int32.to_string x) 
let output_u64 os x = output_string os (Int64.to_string (u64_to_i64 x))
let output_i64 os x = output_string os (Int64.to_string x) 
let output_ieee32 os x = (output_string os "float32 ("; output_string os (Int32.to_string (ieee32_to_bits x)); output_string os ")")
let output_ieee64 os x = (output_string os "float64 ("; output_string os (Int64.to_string (ieee64_to_bits x)); output_string os ")")

let incr_i32 x = int_to_i32 (i32_to_int x+1)
let string_to_i64 x = Int64.of_string x
let string_to_i32 x = Int32.of_string x

(* REVIEW: this is approximate... *)
let string_to_ieee32 x = float_to_ieee32 (float_of_string x)
let u16_to_string x = string_of_int (u16_to_int x)

