// F# Demonstration Scipt
//
// April 2007


// Turn on the lightweight syntax
#light

//---------------------------------------------------------------------------
// Part O. Hello World


System.Console.WriteLine("Hello World")

open Printf

printf "Welcome to Cambridge!\n"

//---------------------------------------------------------------------------
// Part I. Web.


open System.Net
open System
open System.IO

// set up the web proxy
//System.Net.WebRequest.DefaultWebProxy <- System.Net.WebRequest.GetSystemWebProxy()

// ok, now fetch a page.  Create the web request,
// wait for the response, read off the response.
let req = System.Net.WebRequest.Create("http://www.yahoo.com")
let stream = req.GetResponse().GetResponseStream()
let reader = new IO.StreamReader(stream)
let html = reader.ReadToEnd()

html

/// Fetch the contents of a web page
let http(url: string) = 
    let req = System.Net.WebRequest.Create(url) 
    let resp = req.GetResponse()
    let stream = resp.GetResponseStream() 
    let reader = new IO.StreamReader(stream) 
    let html = reader.ReadToEnd()
    resp.Close()
    html

let live   = http("http://www.live.com")
let google = http("http://www.google.com")
let bbc    = http("http://news.bbc.co.uk")
let msft   = http("http://www.microsoft.com")
let nytRSS = http("http://www.nytimes.com/services/xml/rss/nyt/HomePage.xml")

nytRSS

// ----------------------------
// Windows Forms


open System.Windows.Forms
open System.Drawing

let form = new Form(Visible = true, Text = "A Simple F# Form", 
                    TopMost = true, Size = Size(600,600))

let textB = new RichTextBox(Dock = DockStyle.Fill, Text = "F# Programming is Fun!",
                            Font = new Font("Lucida Console",16.0f,FontStyle.Bold),
                            ForeColor = Color.DarkBlue)


form.Controls.Add(textB)

textB.Click.Add(fun evArgs -> printf "Click!\n")


// ----------------------------
// Dumping data to a form

open Microsoft.FSharp.Text.StructuredFormat
let show(x) = 
   let opts = { FormatOptions.Default with 
                   PrintWidth=27; 
                   ShowIEnumerable=true;
                   ShowProperties=true }
   textB.Text <- layout_as_string opts x

let clear() = textB.Clear()
let append(x) = textB.AppendText(x + "\n")

// OK, let's pipeline the data to the form...
//let (|>) x f = f x

(1,2,3) |> show

[ for i in 0 .. 100 -> (i,i*i) ] |> show

// ----------------------------
// Scan RSS for news titles

open System.Xml
open System.Collections
open System.Collections.Generic

let xdoc = new XmlDocument()

nytRSS |> show

xdoc.LoadXml(nytRSS)

[ for n in xdoc.SelectNodes("//title") -> n.InnerText ] |> show

[1;2;3] |> show
[ (1,"1"); (2, "2") ] |> show

// ----------------------------
// Search for URLs in HTML

open System.Text.RegularExpressions

bbc |> show

let linkPat = "href=\s*\"[^\"h]*(http://[^&\"]*)\""

let bbcLinks = Regex.Matches(bbc,linkPat)

// Fetch out the matched strings
{ for m in bbcLinks -> m.Groups.Item(1).Value } |> show

// Put that in a function
let getLinks (txt:string) =  
   { for m in Regex.Matches(txt,linkPat)  
     -> m.Groups.Item(1).Value }

// Collect the links
let collectLinks url = getLinks (try http(url) with _ -> "")

collectLinks "http://www.wired.com" |> show
collectLinks "http://news.google.com" |> show

// ----------------------------
// Crawling (Synchronous)

let limit = 10


let rec crawl (visited: Set<string>) url =
 
    // show the URL...
    append (url + "\n")

    // DoEvents() needed for a synchronous crawl when running in the GUI thread
    Application.DoEvents(); 

    // are we done?
    if visited.Count >= limit or visited.Contains(url) 
    then visited
    
    else 
        // get the results and crawl...
        let links = collectLinks url 
        let visited = visited.Add(url)
        Seq.fold crawl visited links
    
    
clear()

crawl Set.empty "http://news.google.com"

// ----------------------------
// HTTP Requests (Asynchronous)

open System.Threading
open System.Collections.Generic


/// Fetch the URL, and when we're done call the continuation
/// function.  
let httpAsync (url:string) (cont: string -> unit) =
    let req = WebRequest.Create(url) 
    req.BeginGetResponse((fun iar -> 
         using (req.EndGetResponse(iar)) (fun rsp -> 
              let reader = new StreamReader(rsp.GetResponseStream()) 
              reader.ReadToEnd())  
         |> cont), 0) |> ignore

httpAsync  "http://www.microsoft.com" (fun html -> show html)
httpAsync  "http://www.google.com" (fun html -> show html)

/// Fetch the URL, get the links, and when we're done call the continuation
/// function.  
let collectLinksAsync url cont = httpAsync url (getLinks >> cont)
collectLinksAsync "http://www.microsoft.com" (fun urls -> show urls)



// ---------------------------------------------
// Random web walk 

// Random numbers

let rand = new System.Random()
let throwDice xs dflt =
    let xs = Array.of_seq xs 
    let n = Array.length xs 
    if n=0 then dflt else xs.[rand.Next n]

// Web browser control inside a form
open System
open System.Windows.Forms

let wb = new WebBrowser(Dock = DockStyle.Fill, AllowNavigation = true)

let webForm = new Form(Visible = true, Size = Size(600,400), TopMost = true)
webForm.Controls.Add(wb)

// Point it at pages and get the text

wb.Navigate("http://news.bbc.co.uk")
let text = wb.DocumentText

// Regular expressions  

open System.Text.RegularExpressions
let rx   = Regex("http://news.bbc.co.uk/[a-z0-9_.-/]+stm")

// Regular expression to filter urls 

let urlsOfDocument (doc : HtmlDocument) =   
   { for elt in doc.Links 
     let url = elt.GetAttribute("href")
     when rx.IsMatch(url)
     -> url }

let randomLink doc =
  let url  = throwDice (urlsOfDocument doc) "http://news.bbc.co.uk" 
  show (sprintf "JUMP: %s\n" url);
  url

// Click on a timer event

let randomClick () = wb.Navigate(randomLink(wb.Document))
let timer = new Timer(Interval = 1500)
timer.Tick.Add(fun _ -> randomClick ())
timer.Start()

// Enough!

timer.Stop()


// ----------------------------
// Crawling (Asynchronous)


/// Spawn a worker thread that runs the given function
let spawn f = 
    ThreadPool.QueueUserWorkItem(fun _ -> f ()) |> ignore

/// Perform a GUI thread operation from a worker thread
let remote f x = 
    form.Invoke(new MethodInvoker(fun () -> f x)) |> ignore
  
// Shared state (protect by locks!)
let visited = HashSet.Create() 

// Each invocation of this function represents one
// work item in the search.  Results are written to the
// local shared state.
let rec search f url = 
    if not (visited.Contains(url)) && 
        visited.Count < limit
    then 
        lock visited (fun () -> visited.Add(url));
        f url;
        spawn (fun () -> collectLinksAsync url (Seq.iter (search f)))

search (remote append) "http://news.google.com" 



// ------------------------------------------
// Quick Start: pattern matching...

let urlFilter url agent =
  match (url,agent) with 
  | "http://www.control.org", 99 -> true
  | "http://www.kaos.org"   , _  -> false
  | _, 86 -> true
  | _ -> false

urlFilter "http://www.control.org" 86 |> show
urlFilter "http://www.kaos.org" 99 |> show
urlFilter "http://www.smart.org" 86 |> show

// ------------------------------------------
// Quick Start: structured data...


let people = [ ("Adam", None);
               ("Eve" , None);
               ("Cain", Some("Adam","Eve"));
               ("Abel", Some("Adam","Eve")) ]

let showParents (nm,parents) = 
   match parents with  
   | Some(dad,mum) -> sprintf "%s has dad %s, mum %s\n" nm dad mum
   | None          -> sprintf "%s has no parents!\n" nm

clear()

people |> List.iter (showParents >> append)


// ------------------------------------------
// Quick Start: .NET collections...


open System.Collections.Generic

let capitals = new Dictionary<string, string>()

capitals.["Great Britain"] <- "London"
capitals.["France"] <- "Paris"

capitals.ContainsKey("France") |> show

capitals.ContainsKey("Sweden") |> show

capitals.["France"] |> show

capitals.Keys |> show

capitals.Values |> show



// ------------------------------------------
// Quick Start: Nested Functions...

open System.Drawing

let remap (r1: Rectangle) (r2: Rectangle) =

    let scalex = float r2.Width / float r1.Width 
    let scaley = float r2.Height / float r1.Height 
    let scale s i = truncate (float i * s)
    let mapx x = r2.Left + scale scalex (x - r1.Left) 
    let mapy y = r2.Top + scale scaley (y - r1.Top)
    let mapp (p: Point) = new Point(mapx p.X, mapy p.Y) 
    let maps (s: Size) = new Size(mapx s.Width, mapy s.Height) 
    let mapper (r:Rectangle) = new Rectangle(mapp r.Location, maps r.Size)

    //mapx,mapy,mapp,maps,mapper
    mapper


let form3 = new Form(Visible = true, TopMost = true)

/// Now create the specific mapping functions from a (0,0,100,100)
/// rectangle to the current client rectangle
let mapper = remap (new Rectangle(0,0,100,100)) form3.ClientRectangle

let image = Image.FromFile(@"C:\Users\dsyme\Pictures\diekleineraupenimmersatt.jpg")
let g = form3.CreateGraphics()
g.DrawImage(image,mapper (new Rectangle(40,40,20,20)))
g.DrawImage(image,mapper (new Rectangle(0,0,30,30)))
g.DrawImage(image,mapper (new Rectangle(70,70,30,30)))


// ------------------------------------------
// Quick Start: Enumerables

/// A range of integers

seq { 0 .. 10 } |> show

seq { 'a' .. 'z' } |> show


let checkerboardCoordinates n = 
  seq { for row in 0 .. n-1
        for col in 0 .. n-1
        when (row+col) % 2 = 0
        -> (row,col) }

checkerboardCoordinates 5 |> show

let form2 = new Form(TopMost = true, Visible = true, Width = 400, Height = 400)

let g2 = form2.CreateGraphics()
let mapper2 = remap (new Rectangle(0,0,100,100)) form2.ClientRectangle
for p in checkerboardCoordinates 4 do
   let (row,col) = p 
   g2.DrawImage(image,mapper2 (new Rectangle(row*25,col*25,25,25)))


form2.Paint.Add(fun evArgs -> 
    let g2 = evArgs.Graphics
    let mapper2 = remap (new Rectangle(0,0,100,100)) form2.ClientRectangle
    for p in checkerboardCoordinates 4 do
       let (row,col) = p 
       g2.DrawImage(image,mapper2 (new Rectangle(row*25,col*25,25,25)))
)

form2.Resize.Add(fun _ -> form2.Invalidate())

// ------------------------------------------
// Quick Start: Lazy I/O with Enumerables


open System.IO
let rec allFiles dir =
    seq
     { for file in Directory.GetFiles(dir) do 
          yield file 
       for subdir in Directory.GetDirectories dir  do
          for file   in allFiles subdir do 
              yield file }

allFiles @"C:\" |> show
allFiles @"C:\WINDOWS" |> show
allFiles @"C:\WINDOWS" |> Seq.take 100 |> show



// ------------------------------------------
// Quick Start: Making things generic...

let rec hcf a b =
    if   a=0 then b
    elif a<b then hcf a (b-a)
    else          hcf (a-b) b


hcf 24 32

// Now a generic version of the same...
// note the inner function is identical.
let hcfGeneric (zero,(-),(<)) =
    let rec hcf a b =
        if   a=zero then b
        elif a<b then hcf a (b-a)
        else          hcf (a-b) b 
    hcf

let hcfInt    = hcfGeneric (0, (-),(<))
let hcfInt64  = hcfGeneric (0L,(-),(<))
let hcfBigInt = hcfGeneric (0I,(-),(<))

hcfInt 24 32 |> show

hcfInt64 22483098324L 3344230983422L |> show

hcfBigInt 8951387799711704I 89525850428414024I |> show

hcfBigInt 22431105747533235203990810432328I 41918933223213413258167616942323321I |> show


// ------------------------------------------
// Quick Start: Making it generic (b)...

type NumericOps<'a> =
  { Zero: 'a;
    Subtract: 'a -> 'a -> 'a;
    LessThan: 'a -> 'a -> bool; }


let intOps    = { Zero=0 ; Subtract=(-); LessThan=(<) }
let bigIntOps = { Zero=0I; Subtract=(-); LessThan=(<) }
let int64Ops  = { Zero=0L; Subtract=(-); LessThan=(<) }

let hcfGenericB (ops : NumericOps<'a>) =
    let zero = ops.Zero 
    let (-) = ops.Subtract 
    let (<) = ops.LessThan 
    let rec hcf a b =
        if a= zero then b
        elif a < b then hcf a (b-a)
        else hcf (a-b) b 
    hcf

let hcfIntB    = hcfGenericB intOps
let hcfBigIntB = hcfGenericB bigIntOps

hcfIntB 24 32

hcfBigIntB 22431105747535203990810432I 4191892134132581676169464I |> show;;

// ------------------------------------------
// Backup...

module backup = 
    do [html;live;google;bbc;msft;nytRSS] |> List.iteri (fun i x ->
        File.WriteAllText(sprintf @"c:\backup%d.txt" i,x))

    let [html;live;google;bbc;msft;nytRSS] =
        [ for i in 0 .. 5 -> File.ReadAllText(sprintf @"c:\backup%d.txt" i) ]


