/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

/**
 * Framework and base class for all tests.
 * Inherit _doit in your test and use bugIf( boolean )
 * statements for your actual test comparisons.<p>
 *
 * Features:<br>
 * - support for stdout and stderr redirecting<p>
 *
 * Stdout redirection<p>
 *
 * Sometimes you test code which prints something to
 * the console while for the test you don't want to
 * confuse the user, since the test wants to print
 * status information as well.<p>
 * 
 * Console output will be redirected into two byte streams
 * when desired (use 'Test.redirectStandardStreams( true )').
 * The test output will still be printed to the normal
 * standard output streams.<p>
 *
 * When redirection takes place, the test can get a string
 * with the last output to these streams and use this
 * information for testing purposes as well. This buffer
 * can also be cleaned anytime on purpose.
 * <p>
 * 
 * For an example how to write your own tests take a look at
 * the ccl.util.test.UtilTest class.<p>
 *
 * If you write your own test code, extend from this test class
 * and make sure you have these two constructors defined as they
 * are expected from the Test base class in case you want to 
 * build a hierarchy of test classes.
 *
 * <pre>
 *     public MyOwnTest() {
 *         super();
 *     }
 *
 *     public MyOwnTest( Test test ) {
 *         super( test );
 *     }
 * </pre>
 *
 * To execute your test code add for example the following code
 * into your tests static main method:
 *
 * <pre>
 *     public static void main( String[] args ) {
 *         Test test = new MyNewTest();
 *         test.setVerbose( true );
 *         test.setTiming ( true );
 *         test.run();
 *         test.printResult();
 *
 *         System.exit( 0 );
 *     }
 * </pre>
 *
 * So far you have not written any of your own test code and checks. These
 * should go into the protected method '_doIt'.
 * The test base class will invoke that method. Then you put all kinds of
 * statements there which use the
 * 'assert( boolean [, String] )' and 'bugIf( boolean [, String] )' methods
 * of the base test class. Those assertions will be checked and counted.
 * If any assertions fails the test class will print out a message as well
 * as a stack trace. If the succeed a simple '.' will be printed. In the end
 * the total time and the number of successful and total checks will be printed.
 * <p>
 *
 * A successful run of the whole ccl test suite produces the following output:
 *
 * <pre>
 * Testing Main
 * -> Testing Native . SUCCESS 00:00:00.023
 * -> Testing Util
 *    -> Testing text center formatting ...... SUCCESS 00:00:00.007
 *    -> Testing text block formatting ........ SUCCESS 00:00:00.054
 *    -> Testing OutputErrorStreamManager ............ SUCCESS 00:00:00.015
 *    -> Testing SToLAndConcat ............................ SUCCESS 00:00:00.042
 *    -> Testing FileUtil
 *       -> Testing copyDir ... SUCCESS 00:00:00.590
 *       ........................................ SUCCESS 00:00:00.915
 *    -> Testing Init ....................... SUCCESS 00:00:00.591
 *    -> Testing IniFile .... SUCCESS 00:00:00.024
 *    -> Testing ClassPathUtil .. SUCCESS 00:00:00.003
 *    -> Testing SysEnv .... SUCCESS 00:00:00.026
 *    -> Testing Singleton .. SUCCESS 00:00:00.004
 *    ................
 *    -> Testing atoi ................. SUCCESS 00:00:00.003
 *    -> Testing Int/Bytes Conversion ..................................... SUCCESS 00:00:00.005
 *    ................
 *    -> Testing Date Validation .................. SUCCESS 00:00:00.016
 *    ....... SUCCESS 00:00:01.877
 * -> Testing XMLUtil  No Xalan XSLT parser found, skipping test!  SUCCESS 00:00:00.008
 * -> Testing XMLImportHandler  No PostgreSQL database found, skipping tests!  SUCCESS 00:00:00.014
 * -> Testing SwingUtil ..... SUCCESS 00:00:10.969
 * SUCCESS 00:00:13.139
 * *   *   *   *   *   *   *   *   *   *   *   *   *   *
 * ccl.test.MainTest has finished.
 * Hey, kewl, all 251 tests succeeded! :-)
 * </pre>
 * 
 * If you want to build a hierarchy of tests like in the example above, here is an example of how to do
 * it. In your '_doIt' method you invoke another sub test like this:
 *
 * <pre>
 *     protected void _doIt()
 *         throws Exception
 *     {
 *         Test testA = new ASubTest( this );
 *         testA.run();
 *         setTests( testA );
 *
 *         Test testB = new AnotherSubTest( this );
 *         testB.run();
 *         setTests( testB );
 *
 *         // do normal tests below
 *         // ...
 *     }
 * </pre>
 *
 * In the output those two sub tests will then be indented and mentioned at least with their class name
 * and the checks done inside will be added to the number of checks done in your current class.
 *
 * @see ccl.util.test.UtilTest
 *
 * @version 1999-09-03 $Id: Test.java 1.30 2003/11/23 09:13:54 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public abstract class Test implements Runnable 
{
    private long _lGLobalTests = 0;
    private long _lLocalTests = 0;
    private long _lBugs = 0;
    private boolean _bTiming          = false;
    private boolean _bShowLiveSignals = false;
    private boolean _bLastSuccessfull = false;
    private boolean _bLastSubtest     = false;
    private int _subtests = 0;
    private long _lLastBugs = 0;
    private long _lLastLocalTests = 0;
    private String _sLastSubTest = null;
    private Vector _vCalendars = new Vector();

    private static Object _oValue = null;

    private void _showLiveSignal() 
    {
        if ( isVerbose() ) 
        {
            _ensureStandardStreams();
            if ( _bLastSubtest ) 
            {
                Util.println();
            }
            if ( _bLastSubtest || !_bLastSuccessfull ) 
            {
                if ( _subtests > 0 ) 
                {
                    Util.print( Util.getSpaces( (_subtests - 1 ) * 3 ) );
                }
            }
            Util.showLiveSignal();
            _ensureRedirectedStreams();
            _bLastSuccessfull = true;
            _bLastSubtest = false;
        }
    }

    protected Object _getValue() 
    {
        return _oValue;
    }

    /**
     * @param sName_ e.g. "jacob".
     */
    protected void _enterSubTest( String sName_ ) 
    {
        if ( !isVerbose() ) 
        {
            return;
        }

        _ensureStandardStreams();

        if ( _subtests > 0 ) 
        {
            Util.println();
            for( int subtest = 1; subtest < _subtests; subtest++ ) 
            {
                Util.print( "   " );
            }
            Util.print( "-> " );
        }
        else if ( 0 < _lLocalTests ) 
        {
            Util.println();
        }

        _sLastSubTest = sName_;
        _lLastBugs = _lBugs;
        Util.print( "Testing " +
                    sName_ + " " );
        _bLastSubtest = false;
        _bLastSuccessfull = true;

        if ( isTiming() ) 
        {
            _vCalendars.addElement( Util.getCalendar( "UTC" ) );
        }

        _ensureRedirectedStreams();

        _subtests++;
    }
    
    protected void _exitSubTest() 
    {
        if ( !isVerbose() ) 
        {
            return;
        }
        _subtests--;
        _lLastLocalTests = _lLocalTests;

        _ensureStandardStreams();

        if ( _bLastSubtest || !_bLastSuccessfull ) 
        {
            Util.println();
            if ( _subtests > 0 ) 
            {
                Util.print( Util.getSpaces( _subtests * 3 ) );
            }
        }
        else 
        {
            Util.print( " " );
        }
        if ( _lBugs == _lLastBugs )
        {
            Util.print( "SUCCESS" );
        }
        else 
        {
            Util.print( "FAILURE" );
        }
        if ( isTiming() ) 
        {
            Calendar calStart = (Calendar) _vCalendars.
                   lastElement();
            Calendar calEnd = Util.getCalendar( "UTC" );
            long delta = calEnd.getTime().getTime() - 
                   calStart.getTime().getTime();
            Util.debug( this, "_exitSubTest().delta: " + delta );
            Calendar calDelta = Util.getCalendar( "UTC" );
            calDelta.set( Calendar.MILLISECOND, (int) (delta                      % 1000) );
            calDelta.set( Calendar.SECOND     , (int) ((delta / 1000)             % 60  ) );
            calDelta.set( Calendar.MINUTE     , (int) ((delta / (60 * 1000))      % 60  ) );
            calDelta.set( Calendar.HOUR_OF_DAY, (int) ((delta / (60 * 60 * 1000)) % 24  ) );

            calEnd.setTime( new Date( delta ) );
            _vCalendars.removeElementAt( _vCalendars.size() - 1 );

            Util.print( " " + Util.getTime( calDelta ) + "." +
                        Util.paddWithZero( delta % 1000, 3 ) );
        }

        _ensureRedirectedStreams();

        _bLastSubtest = true;
    }

    /**
     * @deprecated   Use setTests(..) instead.
     */
    protected void _setTests(Test pTest_) 
    {
        setTests( pTest_ );
    }

    protected void _increment() 
    {
        _lGLobalTests++;
        _lLocalTests++;
        _showLiveSignal();
    }

    /**
     * @deprecated   Use setVerbose(..) instead.
     */
    protected void _showLiveSignals(boolean bShowLiveSignals_) 
    {
        _bShowLiveSignals = bShowLiveSignals_;
    }
    
    public long getLocalTests() 
    {
        return _lLocalTests;
    }
    
    /**
     * The total number of tests done so far.
     */
    public long getGlobalTests() 
    {
        return _lGLobalTests;
    }

    /**
     * The total number of failures found so far.
     */
    public long getBugs() 
    {
        return _lBugs;
    }

    /**
     * Count one more failure.
     */
    public void setBug() 
    {
        _lBugs++;
    }

    /**
     * @deprecated
     */
    public void setValue(Object oValue_) 
    {
        _oValue = oValue_;
    }

    /**
     * Checks if each test condition handled will be visually presented with a dot on the standard output.
     */
    public boolean isVerbose() 
    {
        return _bShowLiveSignals;
    }

    /**
     * Defines if each test condition handled will be visually presented with a dot on the standard output.
     */
    public void setVerbose( boolean bVerbose_ ) 
    {
        _bShowLiveSignals = bVerbose_;
    }

    /**
     * You want timing information? Set this to true!
     */
    public void setTiming( boolean bTiming_ ) 
    {
        _bTiming = bTiming_;
        if ( isTiming() ) 
        {
            setVerbose( isTiming() );
        }
    }

    /**
     * True if timing information gets printed in addition to the standard test output.
     */
    public boolean isTiming() 
    {
        return _bTiming;
    }

	/**
	 * Prints a summary of all tests to standard output.
	 */
	public void printResult()
	{
		printResult( this );
	}
	
    /**
     * Prints a summary of all tests to standard output.
     */
    public static void printResult(Test pTest_) 
    {
        _ensureStandardStreams();

        Util.println("\n*   *   *   *   *   *   *   *   *   *   *   *   *   *");
        Util.println(pTest_.getClass().getName() + " has finished.");
        if (pTest_.getBugs() == 0) 
        {
            Util.println("Hey, kewl, all " + pTest_.getGlobalTests() +
                         " tests succeeded! :-)");
        }
        else 
        {
            Util.print( "I hate to say it, but " + pTest_.getBugs() +
                        " test" );
            if ( pTest_.getBugs() > 1 ) 
            {
                Util.print( "s" );
            }
            Util.println( " out of " + pTest_.getGlobalTests() +
                          " didn't succeed! :-(");
        }
    }

    private void _processFailure() 
    {
        this.setBug();
        _ensureStandardStreams();
        if ( isVerbose() && _bLastSuccessfull ) 
        {
            Util.println();
        }
        Util.print(this.getClass().getName() + ": ");
        if (_lLocalTests != _lGLobalTests) 
        {
            Util.println("Test " + _lLocalTests + "/" + _lGLobalTests +
                         " failed!");
        }
        else 
        {
            Util.println("Test " + _lLocalTests + " failed!");
        }
        _ensureRedirectedStreams();
    }

    private boolean _check( boolean bCondition_ ) 
    {
        _increment();
        _bLastSubtest = false;

        return bCondition_;
    }

    private void _printStackTrace() 
    {
        _ensureStandardStreams();
        Util.print( "    Stack trace: " );
        try 
        {
            throw new Exception();
        }
        catch( Exception exception ) 
        {
            exception.printStackTrace();
        }
        _ensureRedirectedStreams();
    }

    private void _printMessage( String sDescription ) 
    {
        Vector vLines = Util.stringToLines(sDescription);
        _ensureStandardStreams();
        for(Enumeration e = vLines.elements(); e.hasMoreElements(); ) 
        {
            Util.println("    " + ((String) e.nextElement()));
        }
        _ensureRedirectedStreams();
    }

    private static boolean _bTest = false;

    /**
     * Default constructor which should be overwritten by an implementation of a custom test class.
     */
    public Test() 
    {
        super();
    }
    
    /**
     * If you build a hierarchy of test use this constructor to create a new sub test.
     */
    public Test( Test tstParent_ ) 
    {
        super();

        setParentTest( tstParent_ );
    }

    /**
     * If you build a hierarchy of test use this method to set a parent test object
     * before running the checks of the sub test.
     */
    public void setParentTest( Test tstParent_ )
    {
        setVerbose( tstParent_.isVerbose() );
        setTiming ( tstParent_.isTiming () );

        _subtests = tstParent_._subtests;
    }

    /**
     * The opposite of assert. If the condition is true, it gets counted as a failure.
     */
    public boolean bugIf(boolean bCondition) 
    {
        if ( _check( bCondition ) ) 
        {
            _processFailure();
            _printStackTrace();
        }
        _bLastSuccessfull = !bCondition;
        
        return bCondition;
    }

    /**
     * The opposite of assert. If the condition is true, it gets counted as a failure.
     */
    public boolean bugIf(boolean bCondition, String sDescription) 
    {
        if ( _check(bCondition) ) 
        {
            _processFailure();
            _printMessage( sDescription );
            _printStackTrace();
        }
        _bLastSuccessfull = !bCondition;
        
        return bCondition;
    }
    
    /**
     * The opposite of assert. If the condition is true, it gets counted as a failure.
     */
    public boolean bugIf( boolean bCondition, 
                          String sDescription,
                          Throwable pThrowable)
    {
        if ( _check(bCondition) ) 
        {
            _processFailure();
            _printMessage( sDescription );
            _ensureStandardStreams();
            Util.print( "    " );
            pThrowable.printStackTrace();
            _ensureRedirectedStreams();
        }
        _bLastSuccessfull = !bCondition;
        
        return bCondition;
    }

    /**
     * Conduct a check that a given object is not null.
     * Otherwise count as a failed test.
     */
    public void assertNotNull( Object object_ ) 
    {
        bugIf( object_ == null, "Assert not null test failed!" );
    }
    
    /**
     * Conduct a check that a given object is not null.
     * Otherwise count as a failed test.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertNotNull( Object object_
                               , String sMessage_ )
    {
        bugIf( object_ == null, sMessage_ );
    }
    
    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     *
     * @deprecated   use 'Assert' instead since as of jdk release 1.4
     *               'assert' is a keyword.
     */
    public void assert( boolean bCheck_ ) 
    {
        bugIf( bCheck_ == false, "Assert test failed!" );
    }
    
    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     *
     * @deprecated   use 'Assert' instead since as of jdk release 1.4
     *               'assert' is a keyword.
     */
    public void assert( boolean bCheck_
                        , String sMessage_ )
    {
        bugIf( bCheck_ == false, sMessage_ );
    }
    
    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     */
    public void Assert( boolean bCheck_ ) 
    {
        bugIf( bCheck_ == false, "Assert test failed!" );
    }
    
    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void Assert( boolean bCheck_
                        , String sMessage_ )
    {
        bugIf( bCheck_ == false, sMessage_ );
    }
    
    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     */
    public void assertTrue( boolean bCheck_ ) 
    {
        bugIf( bCheck_ == false, "Assert test failed!" );
    }
    
    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertTrue( boolean bCheck_
                            , String sMessage_ )
    {
        bugIf( bCheck_ == false, sMessage_ );
    }

    /**
     * Conducts a check that a given object fullfills a 
     * given condition.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     * This message is for (local :-) conformance with the JUnit Assert API.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertTrue( String sMessage_ 
                            , boolean bCheck_ )
    {
        bugIf( bCheck_ == false, sMessage_ );
    }

    /**
     * Conducts a check that a given boolean equals another expected boolean value.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     * This message is for (local :-) conformance with the JUnit Assert API.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertEquals( String message, boolean expected, boolean actual )
    {
        Assert( expected == actual, message );
    }

    /**
     * Conducts a check that a given boolean equals another expected boolean value.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     * This message is for (local :-) conformance with the JUnit Assert API.
     */
    public void assertEquals( boolean expected, boolean actual )
    {
        Assert( expected == actual );
    }

    /**
     * Conducts a check that a given integer equals another expected integer value.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     * This message is for (local :-) conformance with the JUnit Assert API.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertEquals( String message, int expected, int actual )
    {
        Assert( expected == actual, message );
    }

    /**
     * Conducts a check that a given string equals another expected string value.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     * This message is for (local :-) conformance with the JUnit Assert API.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertEquals( String message, String expected, String actual )
    {
        if ( expected == null )
        {
            Assert( actual == expected
                    , message );
        }
        Assert( expected.equals( actual ), message );
    }

    /**
     * Conducts a check that a given string equals another expected string value.
     * If this condition is not fullfilled (boolean is false)
     * this check gets counted as a failure.
     * This message is for (local :-) conformance with the JUnit Assert API.
     *
     * @param   sMessage_   The message gets printed when the
     *                      assertion fails.
     */
    public void assertEquals( String expected, String actual )
    {
        if ( expected == null )
        {
            Assert( actual == expected
                    , "Given string '" + actual + "' is not null as expected" );
        }
        String message = "Given string '" 
            + actual 
            + "' does not have the expected value '" 
            + expected
            + "'";
        Assert( expected.equals( actual ), message );
    }

    /**
     * Conducts a check that a given double valud equals another expected double
     * value inside or equal to a given delta value.
     * If this condition is not fullfilled this check gets counted as a failure.
     * <p>
     * This message does not conform to the JUnit API as it has the order of the
     * expected and actual value object just the other way around, thought that
     * should not make a difference.
     */
    public void assertEquals( double actual, double expected, double delta )
    {
        Assert( Math.abs( actual - expected ) <= Math.abs( delta ) );
    }

    /**
     * Conducts a check that a given double valud equals another expected double
     * value inside or equal to a given delta value.
     * If this condition is not fullfilled this check gets counted as a failure.
     * <p>
     * This message does not conform to the JUnit API as the message string
     * is the last parameter and not the first.
     */
    public void assertEquals( double actual, double expected, double delta, String message )
    {
        Assert( Math.abs( actual - expected ) <= Math.abs( delta ), message );
    }

    /**
     * Increases the test counter as well as the bug counter by one and prints
     * the given message.
     * Methods 'bug' and 'fail' are equivalent.
     *
     * @see #bug
     */
    public void fail( String message )
    {
        bug( message );
    }


    /**
     * Increases the test counter as well as the bug counter by one.
     * Methods 'bug' and 'fail' are equivalent.
     *
     * @see #bug
     */
    public void fail()
    {
        Assert( false );
    }

    /**
     * Increases the test counter as well as the bug counter by one and prints
     * the given message.
     * Methods 'bug' and 'fail' are equivalent.
     *
     * @see #fail
     */
    public void bug( String message )
    {
        bugIf( true, message );
    }
    
    /**
     * Inherit this method to do your test inside of it.
     *
     * @exception   Exception   Whatever can go wrong.
     */
    protected abstract void _doIt() throws Exception;
    
    public void run() 
    {
        if ( isVerbose() ) 
        {
            _enterSubTest( getName() );
        }
        try 
        {
            // hier wird die arbeitet gemacht und
            // diese Fkt. soll auch berschrieben werden.
            _doIt();
        }
        catch( OutOfMemoryError memoryError )
        {
            
            bugIf( true , "Test memory error (" + Util.formatMemoryInfo() + ")!", memoryError );
        }
        catch( Throwable throwable ) 
        {
            bugIf( true , "Test error!", throwable );
        }
        if ( isVerbose() ) 
        {
            _exitSubTest();
        }
    }

    /**
     * The application name of this test object. By default it is the class name
     * without any Test' part in it.
     */
    public String getName()
    {
        return getTestName( this );
    }
 
    /**
     * Given a test object it returns a string containing a name representing
     * that test. For example class ccl.util.UtilTest will result in "Util".
     */
    static public String getTestName( Object testObject )
    {
       String sTestName = Util.getObjectName( testObject );

       return getTestName( sTestName );
    }
    /**
     * Given a test object it returns a string containing a name representing
     * that test. For example class ccl.util.UtilTest will result in "Util".
     *
     * @param   sTestName   name of a test class, e.g. "UtilTest".
     */
    static public String getTestName( String sTestName )
    {
        if ( sTestName.startsWith( "Test" ) )
        {
            return sTestName.substring( 4 );
        }

        int testIndex = sTestName.lastIndexOf( "Test" );
        if ( testIndex == -1 ) 
        {
            testIndex = sTestName.length();
        }

        return sTestName.substring( 0, testIndex );
    }

    /* *
     * A class which extends this Test class can have
     * a static main method as a starting point to
     * execute its test code.
     * <p>
     *
     * Example code for a real test main method:
     * <p>
     *
     * <pre>
     * Test pTest = (Test)(new SomeTest());
     * pTest.initialize( args );
     * pTest.setVerbose( true );
     * pTest.run();
     * printResult( pTest );
     *
     * System.exit( 0 );
     * </pre>
     * /
    public static void main(String[] argv) 
    {
        System.exit( 0 );
    }
    */

    /**
     * Process the arguments from the main class here.
     * This way you can pass arguments to this test also
     * from the test center, which does not use the
     * main method to invoke a test.
     */
    public void initialize( String[] asArg_ ) 
    {
        Util.debug( this, "initialize(..).args: " +
                    Util.toString( Util.objectsToVector
                                   ( asArg_ ) ) );
    }

    public static boolean isTest() 
    {
        return _bTest;
    }

    public static Object getValue() 
    {
        return _oValue;
    }

    /**
     * Returns the directory of the class on which this method is
     * invoked.
     *
     * @see          #getTestDirectory()
     * @deprecated   Use getTestDirectory instead as this method
     *               will not work well together with jar files.
     */
    public String getTestClassDirectory() 
    {
        return FileUtil.getClassPath( this );
    }

    /**
     * This method returns the full path of a dedicated test directory
     * under the application directory.
     * 
     * @return   a string for a director with test data for the
     *           current application, 
     *           e.g. "/home/clemens/src/java/jacob/test".
     */
    public String getTestDirectory() 
    {
        return FileUtil.concatPath( ClassPathUtil.getApplicationHome( this )
                                    , "test" );
    }

    public String toString() 
    {
        /*return "Result for " + getClass().getName() + ":";*/
        return ">> no comment available <<";
    }

    public void setTests( Test test_ ) 
    {
        if (test_ == null) 
        {
            _lGLobalTests = 0;
            _lBugs = 0;
            _lLocalTests = 0;
        }
        else 
        {
            _lGLobalTests += test_.getGlobalTests();
            _lBugs += test_.getBugs();
            _bLastSubtest = true;
            //_lLocalTests++;
            //_showLiveSignal();
        }
    }

    /**
     * Return a one line comment for this test.
     * Should be overwritten be test classes inheriting
     * from this class.
     */
    public String getComment() 
    {
        return ">> no comment available <<";
    }

    private static boolean _bRedirectStandardStreams = false;

    private static PrintStream _oldOut = null;
    private static PrintStream _newOut = null;
    private static PrintStream _oldErr = null;
    private static PrintStream _newErr = null;

    /**
     * When we need to switch between two std output streams,
     * this method helps doing so. E.g. we redirect
     * std out but want to print a "test SUCCESS" message,
     * then we use _setOutOld, print, _setOutNew methods.
     */
    private static void _setOutNew() 
    {
        System.setOut( _newOut );
        System.setErr( _newErr );
    }

    /**
     * When we need to switch between two std output streams,
     * this method helps doing so. E.g. we redirect
     * std out but want to print a "test SUCCESS" message,
     * then we use _setOutOld, print, _setOutNew methods.
     */
    private static void _setOutOld() 
    {
        System.setOut( _oldOut );
        System.setErr( _oldErr );
    }

    /**
     * If standard streams are redirected, normal streams
     * are used now.
     */
    private static void _ensureStandardStreams() 
    {
        if ( _bRedirectStandardStreams ) 
        {
            _setOutOld();
        }
    }

    /**
     * If standard streams are redirected, ensure that
     * redirected streams are in place.
     */
    private static void _ensureRedirectedStreams() 
    {
        if ( _bRedirectStandardStreams ) 
        {
            _setOutNew();
        }
    }

    /**
     * If standard streams are redirected, redirection
     * is established again.
     */

    /**
     * Redirect stdout and stderr into private streams
     * so the do not confuse the user with test output.
     *
     * Look at the class documentation at the top for
     * detailed stdout redirection documentation.
     */
    public static void redirectStandardStreams( boolean bRedirect_ ) 
    {
        if ( _bRedirectStandardStreams == bRedirect_ ) 
        {
            return;
        }

        _bRedirectStandardStreams = bRedirect_;

        if ( _bRedirectStandardStreams ) 
        {
            if ( _newOut == null ) 
            {
                // redirect standard output
                _oldOut = System.out;
                ByteArrayOutputStream byteStream      = new ByteArrayOutputStream();
                _newOut = new PrintStream( byteStream );
            
                // redirect standard error
                _oldErr = System.err;
                ByteArrayOutputStream errorByteStream = new ByteArrayOutputStream();
                _newErr = new PrintStream( errorByteStream );
            }
            _setOutNew();
        }
        else 
        {
            _setOutOld();
        }
    }
}
