/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.FileUtil;
import ccl.util.Init;
import ccl.util.Util;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.MouseAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.net.URL;
import java.util.MissingResourceException;
import java.util.Vector;
import javax.help.HelpSet;
import javax.help.HelpBroker;
import javax.swing.JFrame;
import javax.swing.JMenuItem;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.ColorUIResource;
import javax.swing.plaf.metal.DefaultMetalTheme;
import javax.swing.plaf.metal.MetalLookAndFeel;
import javax.swing.plaf.metal.MetalTheme;

/**
 * MainJFrame can be used to inherit you applications main
 * frame class from, which would act as a controller and view
 * of your application.
 *
 * @version  $Id: MainJFrame.java,v 1.29 2003/05/01 16:44:22 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class MainJFrame extends ExitJFrame
                        implements Runnable,
                                   ActionListener,
                                   ChangeListener
{
    /**
     * Directory used by default for documentation.
     * Ccl lib uses 'doc' for this, thought more and more
     * projects seem to choose 'docs' instead.
     */
    public static final String S_DOC_DIR = "doc";
    private static final String S_MODIFIED_TITLE = " *";

    private Thread _pThread = null;
    private RunnableAction _pRunnableAction = null;
    private boolean _bStopThread = false;
    private boolean _bExit = false;
    private boolean _bAboutSelected = false;
    private boolean _bModified = false;

    private Init _pInit = null;

    private boolean _bBusy = false;

    /**
     * Turns on the hour glass mouse cursor and disables all
     * mouse and keyboard events of the main frame.
     */
    public void setBusy( boolean busy )
    {
        _bBusy = busy;
        getGlassPane().setVisible( busy );
        int cursorType = Cursor.DEFAULT_CURSOR;
        if ( busy )
        {
            cursorType = Cursor.WAIT_CURSOR;
        }
        getGlassPane().setCursor( Cursor.getPredefinedCursor( cursorType ) );
    }

    /**
     * Returns true if the application is busy with some task like
     * saving right now.
     * This has some impact, for example exit will not work while
     * the application is busy.
     */
    public boolean isBusy()
    {
        return _bBusy;
    }
    
    public MainJFrame() 
    {
        super();

        // needed later for setting hour glass mouse cursor
        // and using the glass pane to interrupt input events
        getGlassPane().addMouseListener( new MouseAdapter() { } );
        getGlassPane().addKeyListener  ( new KeyAdapter  () { } );
    }
    
    public MainJFrame(String sTitle_) 
    {
        this(sTitle_, new String[0], "");
    }
    
    public MainJFrame(String[] asArg_, String sRcsHeader_) 
    {
        this("", asArg_, sRcsHeader_);
    }
    
    public MainJFrame(String sTitle_, String[] asArg_,
                      String sRcsHeader_)
    {
        this();

        initialize( sTitle_, asArg_, sRcsHeader_ );
    }

    public MainJFrame(String sTitle_, 
                      String[] asArg_,
                      String sRcsHeader_,
                      String sIniFile_)
    {
        this();

        initialize( sTitle_, asArg_, sRcsHeader_, sIniFile_ );
    }

    public void initialize( String[] asArg_, String sInfoHeader_ ) 
    {
        initialize( "", asArg_, sInfoHeader_ );
    }

    public void initialize( String sTitle_,
                            String[] asArg_, 
                            String sRcsHeader_ ) 
    {
        initialize( sTitle_, asArg_, sRcsHeader_, null );
    }
    
    public void initialize( String   sTitle_,
                            String[] asArg_, 
                            String   sRcsHeader_,
                            String   sIniFileContent_ ) 
    {
        Util.debug( this, "initialize(..).START");

        setTitle( sTitle_ );

        setDefaultCloseOperation( JFrame.DO_NOTHING_ON_CLOSE );
        _bExit = false;
        
        _pInit = new Init(this, asArg_, sRcsHeader_, sIniFileContent_ );
        this.setBackground(_pInit.getBackground());
        this.setForeground(_pInit.getForeground());
        
        String sIconFileName = _pInit.getApplicationName().
               toLowerCase() + ".gif";
        Util.debug( "sIconFileName: " + sIconFileName );
        try 
        {
            URL urlImage = this.getClass()
                               .getResource( sIconFileName );
            Util.debug( this
                        , "initialize(..).urlImage: "
                          + urlImage );
            if ( urlImage != null ) 
            {
                Image pImage = Toolkit.getDefaultToolkit()
                                      .getImage( urlImage );
                setIconImage( pImage );
            }
            else 
            {
                Util.printlnErr( "Warning: could not find image for '"
                                 + _pInit.getApplicationName()
                                 + "' application main frame." );
            }
        }
        catch( MissingResourceException pMissingResourceException ) 
        {
            Util.printlnErr( this
                             , "initialize(..)"
                               + ".pMissingResourceException for file "
                               + sIconFileName 
                               + ": " 
                               + pMissingResourceException );
        }
        catch( NullPointerException pNullPointerException ) 
        {
            Util.printlnErr( this
                             , "initialize(..).pNullPointerException for file "
                               + sIconFileName );
            pNullPointerException.printStackTrace();
        }
        
        SwingUtil.setMainFrame(this);
    }
    
    public synchronized void setMenuBar(Vector vMenus_) 
    {
        MainJMenuBar pMainJMenuBar = new MainJMenuBar(vMenus_, this);
        Util.debug( this, "setJMenuBar(..).pMainJMenuBar: " + pMainJMenuBar );
        /*pMainJMenuBar.setBackground(_pInit.getBackground());*/
        setJMenuBar(pMainJMenuBar);
    }
    
    public synchronized void updateMenu() 
    {
        // Alte Files
        ((MainJMenuBar) getJMenuBar()).updateMenu();
    }
    
    public void runAction( RunnableAction pRunnableAction_ ) 
    {
        if (_pThread != null && pRunnableAction_ != null && 
            _pRunnableAction == null) 
        {
            _pRunnableAction = pRunnableAction_;
            _pThread.interrupt();
        }
    }

    protected void runHook() 
    {
        // nur zum berladen
        if ( _pRunnableAction != null ) 
        {
            try 
            {
                _pRunnableAction.run();
            }
            catch(Exception pException) 
            {
                pException.printStackTrace();
            }
            _pRunnableAction = null;
        }
    }
    
    public Init getInit() 
    {
        return _pInit;
    }
    
    public void run() 
    {
        _bStopThread = false;
        while(!_bStopThread && _pThread != null) 
        {
            // fr Hook.
            runHook();
            
            if (_bAboutSelected) 
            {
                _bAboutSelected = false;
                /*ccl.awt.AWTUtil.showAboutDialog( this, getInit() );*/
                ccl.swing.SwingUtil.showAboutDialog( this );
                /*requestFocus();*/
            }
            
            if (_bExit) 
            {
                _bExit = exit();
                _bStopThread = _bExit;
                continue;
            }
            
            // if nothing else to do, go to sleep
            try 
            {
                _pThread.sleep(500);
            }
            catch (InterruptedException e) 
            {
            }
        }
    }
    
    public void start() 
    {
        Util.debug( this, "start().START");
        if (_bExit) 
        {
            return;
        }
        
        while (_pThread != null) 
        {
            stop();
            _pThread.interrupt();
            try 
            {
                _pThread.join();
                _pThread = null;
            }
            catch(InterruptedException e) 
            {
            }
        }
        _pThread = new Thread(this);
        _pThread.start();

      boolean bFinished = false;
      while (bFinished == false) 
      {
          try 
          {
              _pThread.join();
              bFinished = true;
          }
          catch(InterruptedException ie) 
          {
          }
      }
      Util.debug("MainJFrame.start().END");
      /*Util.debug("MainJFrame.start().1");
        if (_bExit) {
        Util.debug("MainJFrame.start().2");
        return;
        }
        if (_pThread != null) {
        stop();
        }
        _pThread = new Thread(this);
        _pThread.start();
        Util.debug("MainJFrame.start().3");
        boolean bFinished = false;
        while (bFinished == false) {
        try {
        Util.debug("MainJFrame.start().4");
        _pThread.join();
        bFinished = true;
        } catch(InterruptedException ie) {
        ;
        }
        }
        Util.debug("MainJFrame.start().5");*/
    }
    
    public void stop() 
    {
        _bStopThread = true;
        if (_pThread != null) 
        {
            _pThread.interrupt();
        }
        /*if (_pThread != null) {
          _pThread = null;
          }*/
    }
    
    /*public void exit() {
      Util.debug("MainJFrame: exit");
      stop();
      super.exit();
      }*/
    
    // respond to the Exit menu choice
    public void actionPerformed(ActionEvent pActionEvent_) 
    {
        Util.debug("MainJFrame: actionPerformed: pActionEvent_: " +
                   pActionEvent_);
        Object oSource = pActionEvent_.getSource();
        if (oSource instanceof JMenuItem) 
        {
            String sMenuItem = ((JMenuItem) oSource).getText();
            if ( sMenuItem.equals( "Beenden" ) 
                 || sMenuItem.equals("Exit") ) 
            {
                Util.debug("MainJFrame.actionPerformed(..).EXIT");
                setExit();
            } 
            else if ( sMenuItem.equals("Info...") 
                      || sMenuItem.equals("About...") 
                      || sMenuItem.equals("Info") 
                      || sMenuItem.equals("About") )
            {
                _bAboutSelected = true;
            } 
            else if ( sMenuItem.equals("Inhalt...") 
                      || sMenuItem.equals("Contents...") 
                      || sMenuItem.equals("Inhalt") 
                      || sMenuItem.equals("Contents") )
            {
                // do nothing, help set is already handling it itself
                /*showHelpContents();*/
            } 
            else if ('1' <= sMenuItem.charAt(0) && sMenuItem.charAt(0) <= '9') 
            {
                openOldProject(sMenuItem.substring(sMenuItem.indexOf(' ') + 1,
                                                   sMenuItem.length()));
                interrupt();

                return;
            }
        }
    }

    /**
     * Returns the Java Help broker object. This can be used
     * to register a Java Help broker object in this main frame.
     */
    public HelpBroker getHelpBroker() 
    {
        MainJFrame pMainJFrame_ = this;
        HelpSet pHelpSet = null;
        String  sHSFile  = pMainJFrame_.getInit()
               .getHelpBrokerURL();
        try 
        {
            URL urlHelpSet = new URL( sHSFile );
            Util.debug( MainJMenuBar.class.getName() + ".getHelpBroker(..).urlHelpSet: " +
                        urlHelpSet );
            Util.debug( MainJMenuBar.class.getName() + ".getHelpBroker(..).TEST" );
            pHelpSet = new HelpSet( null, urlHelpSet );
            Util.debug(  MainJMenuBar.class.getName() + ".getHelpBroker(..).pHelpSet: " + pHelpSet );
            /*sHSFile = pMainJFrame_.getClass().getName();
              int lastDot = sHSFile.lastIndexOf( '.' );
              if ( lastDot == -1 ) {
              sHSFile = "/doc";
              } else {
              sHSFile = "/" 
              + sHSFile.substring( 0, lastDot )
              .replace( '.', '/' );
              }
              sHSFile += pMainJFrame_.getInit()
              .getApplicationName()
              .toLower()
              + ".hs";
              pHelpSet = HelpSet.findHelpSet( null, sHSFile );*/
        }
        catch( Exception pException ) 
        {
            Util.printlnErr( "HelpSet " + sHSFile +
                             " not found!" );
        }
        HelpBroker pHelpBroker = pHelpSet.createHelpBroker();
        
        return pHelpBroker;
    }

    public void setExit() 
    {
        if ( isBusy() )
        {
            SwingUtil.showMessage( this
                                   , "Application is busy right now, try to exit in a view seconds again.\nThanks." );

            return;
        }

        if ( !_confirmClose() ) 
        {
            return;
        }

        Util.debug("MainJFrame.setExit().BEGIN");
        _bExit = true;
        if (_pThread != null) 
        {
            _pThread.interrupt();
        }
    }

    protected boolean isExitSet() 
    {
        return _bExit;
    }
    
    public void windowClosing( WindowEvent event )
    {
        Util.debug("MainJFrame.windowClosing(..).BEGIN");
        setExit();
    }

    /**
     * A class representing a swing theme used by default for the
     * ccl library. It nails down to a monochrome color scheme
     * using the default system background color.
     */
    public static class CCLMetalTheme extends DefaultMetalTheme 
    {
        private ColorUIResource _secondary3 = null;
        private ColorUIResource _highlight  = null;
        private ColorUIResource _shadow     = null;
        private ColorUIResource _darkShadow = null;

        public CCLMetalTheme( Color color_ ) 
        {
            super();

            _secondary3 = new ColorUIResource( color_                   );
            _highlight  = new ColorUIResource( color_.brighter()        );
            _shadow     = new ColorUIResource( color_.darker()          );
            _darkShadow = new ColorUIResource( color_.darker().darker() );
        }

        public ColorUIResource getControlHighlight() 
        { 
            return _highlight; 
        }  

        public ColorUIResource getControlShadow() 
        { 
            return _shadow; 
        }  

        public ColorUIResource getControlDarkShadow() 
        { 
            return _darkShadow; 
        }  

        protected ColorUIResource getPrimary1() 
        {
            return getSecondary1(); 
        }  // these are blue in Metal Default Theme

        protected ColorUIResource getPrimary2() 
        {
            return getSecondary2(); 
        }

        protected ColorUIResource getPrimary3() 
        {
            return getSecondary3(); 
        };

        protected ColorUIResource getSecondary1() 
        {
            return _darkShadow;
        }  // these are gray in Metal Default Theme

        protected ColorUIResource getSecondary2() 
        {
            return _shadow; 
        }

        protected ColorUIResource getSecondary3() 
        { 
            return _secondary3; 
        }

        public ColorUIResource getPrimaryControlDarkShadow() 
        {
            return _darkShadow;
        }

        public ColorUIResource getPrimaryControlHighlight() 
        {
            return _highlight;
        }

        /*private static ColorUIResource white 
          = new ColorUIResource( 0, 155, 255 );
        */

        //protected ColorUIResource getWhite() { return white; }

        //public ColorUIResource getMenuBackground()
        /*public ColorUIResource getSeparatorForeground() {
            return _highlight;
            }*/

        public ColorUIResource getSeparatorBackground() 
        {
            return getControlHighlight();
        }

        public ColorUIResource getMenuSelectedForeground() 
        {
            return getControlHighlight();
        }

        /*public ColorUIResource getControlInfo() {
            return getControlShadow();
            }*/
    }

    public void setBackground( Color pColor_ ) 
    {
        super.setBackground(pColor_);

	if ( UIManager.getLookAndFeel().getClass().getName().startsWith( "javax.swing.plaf." ) )
	{
	    getRootPane().setBackground(pColor_);
	    getContentPane().setBackground(pColor_);
	    getLayeredPane().setBackground(pColor_);
	    
	    MetalTheme theme = new CCLMetalTheme( pColor_ );
	    MetalLookAndFeel.setCurrentTheme( theme );
	    try 
	    {
		UIManager.setLookAndFeel( new MetalLookAndFeel() );
	    }
	    catch( UnsupportedLookAndFeelException lafException ) 
	    {
	    }

	    //UIDefaults pUIDefaults = UIManager.getLookAndFeelDefaults();
	    //pUIDefaults.put( "control", new ColorUIResource( pColor_ ) );
	    UIManager.put("ScrollBarUI", "ccl.swing.CCLScrollBarUI");
	    //UIManager.put("TableHeader.cellBorder", "ccl.swing.CCLBorder");
	    // thumb is the nibble thing in the middle
	    UIManager.put("ScrollBar.thumb", theme.getControl());
	    UIManager.put("ScrollBar.thumbShadow", theme.getControlShadow());
	    UIManager.put("ScrollBar.thumbHighlight", theme.getControlHighlight());
	    //UIManager.put("ScrollBar.background", theme.getControl());
	    //UIManager.put("ScrollBar.highlight", theme.getControlHighlight());
	    //  UIManager.put("ScrollBar.shadow", theme.getControlShadow());
	    //  UIManager.put("ScrollBar.darkShadow", theme.getControlDarkShadow());
	    
	    //UIDefaults pUIDefaults = UIManager.getLookAndFeelDefaults();
	    //pUIDefaults.put( "control", new ColorUIResource( pColor_ ) );
	    
	    // look in MetalLookAndFeel under defaults what you can set
	    //UIManager.put("Menu.background", pColor_);
	    //UIManager.put("MenuItem.background", pColor_);
	    //UIManager.put("MenuBar.background", pColor_);
	    UIManager.put("TabbedPane.background", pColor_);
	    UIManager.put("TabbedPane.tabAreaBackground", pColor_);
	    UIManager.put("TabbedPane.highlight", pColor_);
	    UIManager.put("TabbedPane.lightHighlight", pColor_);
	    UIManager.put("TabbedPane.selected", pColor_);
	    UIManager.put("TabbedPane.selectedHighlight", pColor_);
	    //UIManager.put("ComboBox.background", pColor_);

	    //UIManager.put( "SplitPane.background", pColor_            );
	    //UIManager.put( "SplitPane.highlight",  pColor_.brighter() );
	    //UIManager.put( "SplitPane.shadow",     pColor_.darker()   );
	    //UIManager.put( "SplitPane.border",     pColor_            );
        
	    //UIManager.put("Slider.background", Color.black);//pColor_);
	    //  UIManager.put("ScrollBar.track", Color.black);//pColor_);
	    //  UIManager.put("ScrollBar.trackHighlight", Color.yellow);//pColor_);


	    
	    //        UIManager.put("Button.background", pColor_);

	    //thumbHighlightColor 
	    //  = UIManager.getColor("ScrollBar.thumbHighlight");
	    //  thumbLightShadowColor 
	    //  = UIManager.getColor("ScrollBar.thumbLightShadow");
	    //  thumbDarkShadowColor 
	    //  = UIManager.getColor("ScrollBar.thumbDarkShadow");
	    //  thumbColor = UIManager.getColor("ScrollBar.thumb");
	    //  trackColor = UIManager.getColor("ScrollBar.track");
	    //  trackHighlightColor 
	    //  = UIManager.getColor("ScrollBar.trackHighlight");
	    //  scrollBarColorsInitialized = true;
	    
	    // foreground stuff should go into its own method
	    UIManager.put( "Label.foreground", Color.black );
	    UIManager.put( "Label.background", pColor_ );
	    UIManager.put( "Panel.background", pColor_ );
	    UIManager.put( "CheckBox.background", pColor_ );
	    UIManager.put( "RadioButton.background", pColor_ );
	    UIManager.put( "TitledBorder.titleColor", Color.black );
	    UIManager.put( "ScrollPane.border", pColor_ );
	    UIManager.put( "ScrollPane.background", pColor_ );

	    // set text background color
	    UIManager.put( "TextArea.background", pColor_ );
	    UIManager.put( "TextPane.background", pColor_ );
	    UIManager.put( "TextArea.selectionBackground", Color.black );
	    UIManager.put( "TextArea.selectionForeground", theme.getControl() );
	    UIManager.put( "TextPane.selectionBackground", Color.black );
	    UIManager.put( "TextPane.selectionForeground", theme.getControl() );
	    UIManager.put( "Tree.background", pColor_ );
	    UIManager.put( "Tree.textBackground", pColor_ );
	    UIManager.put( "Tree.selectionBackground", 
			   pColor_.darker() );
	    UIManager.put( "Tree.selectionForeground", 
			   pColor_.brighter() );
	    UIManager.put( "List.background", pColor_ );
	    UIManager.put( "List.selectionBackground", 
			   pColor_.darker() );
	    UIManager.put( "List.selectionForeground", 
			   pColor_.brighter() );
	    
	    UIManager.put( "TableHeader.background",
			   pColor_ );
	}
    }

    public void interrupt() 
    {
        if (_pThread != null) 
        {
            _pThread.interrupt();
        }
    }

    public void setModified( boolean bModified_ ) 
    {
        _bModified = bModified_;
        if ( _bModified ) 
        {
            // change app title
            String sTitle = getTitle();
            if ( !Util.endsWith( sTitle, S_MODIFIED_TITLE ) ) 
            {
                sTitle += S_MODIFIED_TITLE;
                setTitle( sTitle );
            }
        } 
        else 
        {
            String sTitle = getTitle();
            if ( Util.endsWith( sTitle, S_MODIFIED_TITLE ) ) 
            {
                sTitle = sTitle.substring( 0, sTitle.length() - 
                                           S_MODIFIED_TITLE.
                                           length() );
                setTitle( sTitle );
            }
        }
    }

    public void newFile() 
    {
        try 
        {
            SwingUtil.invokeAndWaitIfNecessary( new Runnable() {
                public void run() {
                    getInit().setFileFullName( "" );
                    setModified( false );
                    setTitle();
                    updateMenu();
                }
            } );
        }
        catch( Exception exception )
        {
            Util.printlnErr( exception );
        }
    }

    /**
     * Interactively open new file/project.
     */
    public void open() 
    {
        if ( !_confirmClose() ) 
        {
            return;
        }

        String sProjectFullName = null;
        Init pInit = getInit();
        String sProjectSuffix = _getProjectSuffix();
        sProjectSuffix = sProjectSuffix.substring( 1 );
        String[] asProjectSuffixes = { sProjectSuffix }; 
        String sOldProjectPath = pInit.getFilePath()
                                      .replace    ( '/', File.separatorChar );
        sProjectFullName = SwingUtil.getFileName
               ( this, 
                 "Choose a Project", 
                 asProjectSuffixes,
                 sOldProjectPath );
        if (sProjectFullName == null) 
        {
            return;
        }
        if ( !FileUtil.existsFile( sProjectFullName ) ) 
        {
            SwingUtil.showMessage( this, "Sorry, the specified file\n\n'" +
                                   sProjectFullName +
                                   "'\n\ndoes not exist!" );

            return;
        }

        /*
        // filename must end with .jiml or .nr
        if ( !Util.endsWith( sProjectFullName, sProjectSuffix ) ) {
            SwingUtil.showMessage( this
                                   , "Warning: Project file\n'"
                                     + sProjectFullName
                                     + "'\ndoes not end with '"
                                     + sProjectSuffix + "'!" );
        }
        // */
        
        open( sProjectFullName );
    }

    /*public void open() {
        setModified( false );
        setTitle();
        updateMenu();
        }*/

    /**
     * If you overwrite this method, it should invoke
     * super.open( filename ) at the end.
     */
    public void open( String sProjectFileName_ ) 
    {
        // history stuff
        getInit().setFileFullName( sProjectFileName_ );
        setModified( false );
        setTitle();
        updateMenu();
    }
    
    public void openOldProject( String sOldProjectName_ ) 
    {
        String sOldProjectFullName = null;
        
        sOldProjectFullName = getInit().getOldFileFullName( sOldProjectName_ );
        
        if ( sOldProjectFullName == null ) 
        {
            // garbage in, we do nothing
            return;
        }
        
        open( sOldProjectFullName );
    }

    protected void _backup( String sFileName_ ) 
    {
        Util.debug( this, "_backup(..).sFileName_: " + sFileName_ );
        Util.debug( this, "_backup(..).exists: " + FileUtil.exists( sFileName_ ) );
        if ( FileUtil.exists( sFileName_ ) ) 
        {
            Util.debug( this, "_backup(..).MOVING" );
            boolean bError = FileUtil.move( sFileName_, sFileName_ + ".bak" );
            Util.debug( this, "_backup(..).error: " + bError );
        }
    }

    public void save() 
    {
        Util.debug( "ccl.swing.MainJFrame.save().ENTER" );
        String sTempFileName = getInit().getFileFullName();
        Util.debug( "MainJFrame.save().sTempFileName: " +
                    sTempFileName );

        if ( Util.isEmpty( sTempFileName ) ) 
        {
            saveAs();

            return;
        }

        _backup    ( sTempFileName );

        save       ( sTempFileName );

        setModified( false         );
        setTitle   ();
        updateMenu ();
    }

    /*public void save() {
        setModified( false );
        setTitle();
        updateMenu();
        }*/

    /**
     * You can overwrite this method to return a default
     * project file suffix even if the ini file is missing.
     */
    protected String _getProjectSuffix() 
    {
        String sProjectSuffix = getInit().getKeyValue
               ( Init.getProjectSuffixKey() );
        if ( Util.isEmpty( sProjectSuffix ) ) 
        {
            sProjectSuffix = ".xml";
            getInit().setKeyValue( getInit().getProjectSuffixKey()
                                   , sProjectSuffix );
        }

        return sProjectSuffix;
    }

    public void saveAs() 
    {
        // find file name
        String sProjectSuffix = _getProjectSuffix();
        String sOldProjectPath = getInit().getFilePath()
                                          .replace('/', File.separatorChar);
        String sFileName = SwingUtil.getFileName
               ( this, 
                 "Save As",
                 "*" + sProjectSuffix,
                 sOldProjectPath,
                 "Save As" );
        if ( sFileName == null) 
        {
            return;
        }

        // filename must end with ".xxx"
        if ( !Util.endsWith( sFileName, sProjectSuffix ) ) 
        {
            SwingUtil.showMessage( this
                                   , "Sorry, project file name must end with '" 
                                     + sProjectSuffix 
                                     + "'." );
            
            return;
        }
        if ( FileUtil.existsDir( sFileName ) ) 
        {
            SwingUtil.showMessage( this
                                   , "Sorry, file\n\n'" 
                                     + sFileName 
                                     + "'\n\nis a directory." 
                                     + "Please choose another file name "
                                     + "to save your data." );

            // try it again
            saveAs();
                
            return;
        }
        if ( FileUtil.existsFile( sFileName ) ) 
        {
            boolean bOK = SwingUtil.isOKOrCancel( this
                                                  , "File\n\n'" 
                                                    + sFileName 
                                                    + "'\n\nexists already. "
                                                    + "Is it OK to overwrite "
                                                    + "it?" );
            if ( !bOK ) 
            {
                return;
            }
        }   
        getInit().setFileFullName( sFileName );

        save();
    }

    /**
     * This is the method the user must implement.
     * User has not to take care of setting saved flag etc.
     */
    public void save( String sFileName_ ) 
    {
    }

    private boolean _bConfirming = false;

    protected boolean _confirmClose() 
    {
        if ( _bConfirming ) 
        {
            // somehow the user managed to ask this question
            // more than once at the same time
            return false;
        }
        _bConfirming = true;

        if (_bModified == false) 
        {
            _bConfirming = false;

            return true;
        }
        int yesNoOrCancel = SwingUtil.wannaSave(this);
        if (yesNoOrCancel == SwingUtil.YES) 
        {
            save();
            _bConfirming = false;

            return true;
        }

        _bConfirming = false;

        return( yesNoOrCancel != SwingUtil.CANCEL );
    }

    /**
     * @deprecated   #_confirmClose
     */
    protected boolean _isQuitGuardOK() 
    {
        return _confirmClose();
    }

    public void stateChanged( ChangeEvent pChangeEvent_ ) 
    {
        setModified( true );
    }

    public void setTitle() 
    {
        // get application name
        String sTitle = getInit().getApplicationName();

        // get project name
        String sProjectName = getInit().getFileName();
        if ( !Util.isEmpty( sProjectName ) ) 
        {
            sTitle += ": " + sProjectName;
        }

        // is modified?
        if ( _bModified ) 
        {
            sTitle += S_MODIFIED_TITLE;
        }

        // set it
        setTitle( sTitle );
    }

    public boolean isModified() 
    {
        return _bModified;
    }
}
