/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.Util;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.JFrame;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.text.Keymap;
import javax.swing.text.PlainDocument;
import javax.swing.text.TextAction;

/**
 * An gui class for retrieving a date.
 *
 * @version $Id: DateField.java,v 1.5 2003/05/01 16:44:14 clemens Exp clemens $
 * @author  Chr. Clemens Lee
 */
public class DateField extends JTextField
                       implements CaretListener
{
    private String _sTimezoneID = null;

    static class DateDocument extends PlainDocument 
    {
        boolean _bInternal = false;

        private String _sTimezoneID = null;

        public DateDocument() 
        {
            super();
        }

        public void setTimezoneID( String sTimezoneID_ ) 
        {
            _sTimezoneID = sTimezoneID_;
        }

        public void remove( int offs_, int length_ ) 
            throws BadLocationException
        {
            if ( _bInternal ) 
            {
                super.remove( offs_, length_ );
            }
        }

        private void _removeChar( int position_ ) 
            throws BadLocationException
        {
            if ( position_ < getLength() ) 
            {
                _bInternal = true;
                super.remove( position_, 1 );
                _bInternal = false;
            }
        }

        private void _insert( int position_, char char_ ) 
            throws BadLocationException
        {
            _removeChar( position_ );
            super.insertString( position_, 
                                Util.cToS( char_ ),
                                null );
        }

        public void insertString( int offs_, 
                                  String sInsert_, 
                                  AttributeSet atsUnused_ ) 
            throws BadLocationException 
        {
            Util.debug( this, "insertString(..).offs_: " +
                        offs_ );
            Util.debug( this, "insertString(..).sInsert_: " +
                        sInsert_ );
            if ( Util.isEmpty( sInsert_ ) ) 
            {
                return;
            }

            String sOldDate = getText( 0, getLength() );
            String sFutureDate = sOldDate.substring( 0, offs_ );
            sFutureDate += sInsert_;
            if ( sOldDate.length() > offs_ + sInsert_.length() ) 
            {
                sFutureDate += sOldDate.substring( offs_ + sInsert_.length() );
            }
            if ( !Util.isDateValid( sFutureDate ) ) 
            {
                if ( sInsert_.length() > 1 || getLength() == 0 ) 
                {
                    // set today
                    String sToday = Util.getDate
                           ( Util.getCalendar( _sTimezoneID ) );
                    _bInternal = true;
                    remove( 0, getLength() );
                    insertString( 0, sToday, atsUnused_ );
                    _bInternal = false;
                }

                return;
            }

            int position = offs_;
            for( ; position < offs_ + sInsert_.length();
                 position++ )
            {
                Util.debug( this, "insertString(..).position: " +
                            position );
                char cNext = sInsert_.charAt( position - offs_ );
                Util.debug( this, "insertString(..).cNext: " +
                            cNext );

                // year
                if ( position < 4 &&
                     Util.isDigit( cNext ) )
                {
                    _insert( position, cNext );
                }

                // '-' in position 4 and 6
                if ( (position == 4 ||
                      position == 7) &&
                     cNext == '-' ) 
                {
                    _insert( position, cNext );
                }

                // month
                {
                    if ( position == 5 ) 
                    {
                        if ( cNext == '0' ) 
                        {
                            _insert( position, cNext );
                        }
                        if ( cNext == '1' )
                        {
                            // not allowed when next digit > 2

                            char cNextMonth = '0';
                            if ( sInsert_.length() > position - offs_ + 1 ) 
                            {
                                cNextMonth = sInsert_.charAt
                                       ( position - offs_ + 1 );
                            }
                            else if ( getLength() > 6 ) 
                            { 
                                cNextMonth = getText( 6, 1 ).charAt( 0 );
                            }

                            if ( cNextMonth <= '2' ) 
                            {
                                _insert( position, cNext );
                            }
                        }
                    }

                    if ( position == 6 ) 
                    {
                        char cMax = '9';
                        boolean bOctoberOrLater = 
                               getText( 5, 1 ).equals( "1" );
                        if ( bOctoberOrLater ) 
                        {
                            cMax = '2';
                        }
                        if ( '0' <= cNext && cNext <= cMax ) 
                        {
                            _insert( position, cNext );
                        }
                    }
                }

                // day
                {
                    if ( position == 8 ) 
                    {
                        if ( '0' <= cNext && cNext <= '2' ) 
                        {
                            _insert( position, cNext );
                        }
                        if ( cNext == '3' )
                        {
                            // not allowed when next digit > 1

                            char cNextDay = '0';
                            if ( sInsert_.length() > position - offs_ + 1 ) 
                            {
                                cNextDay = sInsert_.charAt
                                       ( position - offs_ + 1 );
                            }
                            else if ( getLength() > 9 ) 
                            { 
                                cNextDay = getText( 9, 1 ).charAt( 0 );
                            }

                            if ( cNextDay <= '1' ) 
                            {
                                _insert( position, cNext );
                            }
                        }
                    }

                    if ( position == 9 ) 
                    {
                        char cMax = '9';
                        boolean b30OrGreater = 
                               getText( 8, 1 ).equals( "3" );
                        if ( b30OrGreater ) 
                        {
                            cMax = '1';
                        }
                        if ( '0' <= cNext && cNext <= cMax ) 
                        {
                            _insert( position, cNext );
                        }
                    }
                }
            } // next
            
            if ( position == 4 || position == 7 ) 
            {
                _bInternal = true;
                insertString( position, "-", atsUnused_ );
                _bInternal = false;
            }
        }
    }

    static class DateTextAction extends TextAction 
    {
        private String _sTimezoneID = null;

        public DateTextAction( String sTimezoneID_ ) 
        {
            super( "DateTextAction" );

            _sTimezoneID = sTimezoneID_;
        }
        
        public void actionPerformed( ActionEvent pActionEvent_ ) 
        {
            JTextComponent txtTarget = getTextComponent( pActionEvent_ );

            txtTarget.setText( Util.getDate( Util.getCalendar( _sTimezoneID ) ) );
        }
    }

    static class DeleteTextAction extends TextAction 
    {
        public DeleteTextAction() 
        {
            super( "DeleteTextAction" );
        }
        
        public void actionPerformed( ActionEvent pActionEvent_ ) 
        {
            DateField dateField = (DateField)getTextComponent( pActionEvent_ );

            dateField.clear();
        }
    }

    static class DigitTextAction extends TextAction 
    {
        private boolean _bIncrease = true;

        public DigitTextAction( boolean bIncrease_ ) 
        {
            super( "DigitTextAction" );

            _bIncrease = bIncrease_;
        }
        
        public void actionPerformed( ActionEvent pActionEvent_ ) 
        {
            DateField dateField = (DateField)getTextComponent( pActionEvent_ );

            int caret = dateField.getCaretPosition();
            String sDate = dateField.getText();
            char digit = sDate.charAt( caret );

            if ( digit == '-' ) 
            {
                return;
            }

            char maxDigit = '9';
            if ( caret == 5 ) 
            {
                maxDigit = '1';
            }
            if ( caret == 6 && 
                 sDate.charAt( 5 ) == '1' ) 
            {
                maxDigit = '2';
            }
            if ( caret == 8 ) 
            {
                maxDigit = '3';
            }
            if ( caret == 9 && 
                 sDate.charAt( 8 ) == '3' ) 
            {
                maxDigit = '1';
            }

            if ( _bIncrease ) 
            {
                digit++;
                if ( digit > maxDigit ) 
                {
                    digit = '0';
                    if ( (caret == 6 && 
                          sDate.charAt( 5 ) == '0') ||
                         (caret == 9 &&
                          sDate.charAt( caret -1 ) == '0') ||
                         (caret == 8 &&
                          sDate.charAt( caret + 1 ) == '0') )
                    {
                        digit++;
                    }
                }

                sDate = sDate.substring( 0, caret ) +
                       digit +
                       sDate.substring( caret + 1 );
                
                if ( !Util.isDateValid( sDate ) ) 
                {
                    digit = '0';
                    sDate = sDate.substring( 0, caret ) +
                           digit +
                           sDate.substring( caret + 1 );
                }
            }
            else 
            {
                digit--;
                if ( digit < '0' ) 
                {
                    digit = maxDigit;
                }

                sDate = sDate.substring( 0, caret ) +
                       digit +
                       sDate.substring( caret + 1 );
                
                if ( !Util.isDateValid( sDate ) ) 
                {
                    digit = maxDigit;
                    sDate = sDate.substring( 0, caret ) +
                           digit +
                           sDate.substring( caret + 1 );
                }
            }

                
            if ( Util.isDateValid( sDate ) ) 
            {
                dateField.setText( sDate );
                dateField.setCaretPosition( caret );
            }
        }
    }

    private void _setInternal( boolean internal_ ) 
    {
        ((DateDocument)getDocument())._bInternal = internal_;
    }

    protected Document createDefaultModel() 
    {
        return new DateDocument();
    }

    public DateField( String sTimezoneID_ ) 
    {
        super();

        setBorder( SwingUtil.createCCLBorder() );

        _sTimezoneID = sTimezoneID_;

        ((DateDocument)getDocument()).setTimezoneID( _sTimezoneID );

        // keystrokes
        {
            TextAction pDateTextAction = new DateTextAction
                   ( sTimezoneID_ );
            Keymap kmOld = this.getKeymap();
            Keymap kmNew = this.addKeymap( null, kmOld );
            KeyStroke ksCtrlD = KeyStroke.getKeyStroke
                   ( KeyEvent.VK_D, Event.CTRL_MASK );
            kmNew.removeKeyStrokeBinding( ksCtrlD );
            kmNew.addActionForKeyStroke
                   ( ksCtrlD, pDateTextAction);
            
            KeyStroke ksT = KeyStroke.getKeyStroke
                   ( 't' );
            kmNew.removeKeyStrokeBinding( ksT );
            kmNew.addActionForKeyStroke
                   ( ksT, pDateTextAction);
            
            DeleteTextAction delete = new DeleteTextAction();
            KeyStroke ksDel = KeyStroke.getKeyStroke
                   ( KeyEvent.VK_DELETE, 0 );
            kmNew.removeKeyStrokeBinding( ksDel );
            kmNew.addActionForKeyStroke
                   ( ksDel, delete );
            
            KeyStroke ksBack = KeyStroke.getKeyStroke
                   ( KeyEvent.VK_BACK_SPACE, 0 );
            kmNew.removeKeyStrokeBinding( ksBack );
            kmNew.addActionForKeyStroke
                   ( ksBack, delete );


            DigitTextAction increase = new DigitTextAction( true );
            KeyStroke ksUp = KeyStroke.getKeyStroke
                   ( KeyEvent.VK_UP, 0 );
            kmNew.removeKeyStrokeBinding( ksUp );
            kmNew.addActionForKeyStroke
                   ( ksUp, increase );

            DigitTextAction decrease = new DigitTextAction( false );
            KeyStroke ksDown = KeyStroke.getKeyStroke
                   ( KeyEvent.VK_DOWN, 0 );
            kmNew.removeKeyStrokeBinding( ksDown );
            kmNew.addActionForKeyStroke
                   ( ksDown, decrease );

            this.setKeymap( kmNew );
        }

        addCaretListener( this );

        Dimension dimSize = new Dimension( 92, 24 );
        setPreferredSize( new Dimension( dimSize ) );
        setMinimumSize  ( new Dimension( dimSize ) );
    }

    public void clear() 
    {
        _setInternal( true );
        setCaretPosition( 0 );
        super.setText( "" );
        _setInternal( false );
    }

    public void setToday() 
    {
        setText( Util.getDate( Util.getCalendar( _sTimezoneID ) ) );
    }

    public void setText( String sText_ ) 
    {
        if ( Util.isEmpty( sText_ ) ) 
        {
            clear();

            return;
        }

        super.setText( sText_ );
    }

    private int _oldDot = 0;

    public void caretUpdate( CaretEvent event_ ) 
    {
        int dot = event_.getDot();

        Util.debug( this, "caretUpdate(..).dot:     " + dot     );
        Util.debug( this, "caretUpdate(..)._oldDot: " + _oldDot );

        if ( dot == 10 ) 
        {
            _oldDot = dot;
            this.setCaretPosition( dot -1 );
        }

        if ( ((DateDocument)getDocument())._bInternal ) 
        {
            _oldDot = dot;
            
            return;
        }

        if ( dot == 4 || dot == 7 ) 
        {
            if ( _oldDot > dot ) 
            {
                _oldDot = dot;
                this.setCaretPosition( dot - 1 );
            }
            else if ( getDocument().getLength() > dot ) 
            {
                _oldDot = dot;
                this.setCaretPosition( dot + 1 );
            }
        }
        else 
        {
            _oldDot = dot;
        }
    }

    public boolean isDateValid() 
    {
        return Util.isDateValid( getText() );
    }

    public static void main( String[] asArg_ ) 
    {
        Util.setDebug( true );

        JFrame frmMain = new JFrame( "DateField Test" );
        frmMain.addWindowListener( new WindowAdapter() {
                public void windowClosing( WindowEvent event_ ) 
                {
                    System.exit( 0 );
                }
            } );
        
        DateField pDateField = new DateField( "CET" );
        try 
        {
            //pDateField.getDocument().insertString( 0, "1999-11", null );
        }
        catch( Exception exception ) 
        {
        }
        frmMain.getContentPane().add( pDateField );

        frmMain.setSize( 200, 60 );
        frmMain.setVisible( true );
        SwingUtil.centerComponent( frmMain );
    }
}
