//
//  HMInfoBar.m
//  HMInfoBar
//
//  Created by MIURA Kazki on 10.04.06.
//  Copyright 2006 MIURA Kazki. All rights reserved.
//

#import "HMInfoBar.h"

static const float HMInfoBarMarginX = 12.0;
static const float HMInfoBarClipButtonMarginX = 0.0;


@interface HMInfoBar (Private)
- (void)_setButtons:(NSArray*)buttons toHidden:(BOOL)flag;
- (void)_layoutButton:(HMInfoBarButton*)button inSpace:(NSRect*)spaceFrame;
- (void)_layoutClipButton;
@end

#pragma mark -


@implementation HMInfoBar

//--------------------------------------------------------------//
#pragma mark	Creating Instances
//--------------------------------------------------------------//

- (id)initWithFrame:(NSRect)frame
{
	if (self = [super initWithFrame:frame]) {
		// Initialize
		_buttons = [[NSMutableArray array] retain];
		_marginX = HMInfoBarMarginX;
		_selectedButton = nil;
		_bottomLineColor = [[NSColor windowFrameColor] retain];
		_clipButtonIndex = 0;
		
        // Get bundle
        NSBundle*   bundle;
        bundle = [NSBundle bundleForClass:[self class]];
        
		// Create clip indicator button
		NSImage *clipImage;
		NSRect frame;
		clipImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"clipIndicator"]] autorelease];
		frame.origin = NSZeroPoint;
		frame.size = [clipImage size];
		_clipButton = [[HMMenuButton alloc] initWithFrame:frame];
		[_clipButton setButtonType:NSMomentaryChangeButton];
		[_clipButton setBezelStyle:NSRegularSquareBezelStyle];
		[_clipButton setAutoresizingMask:NSViewNotSizable];
		[_clipButton setBordered:NO];
		[_clipButton setImage:clipImage];
		[_clipButton setFont:[NSFont systemFontOfSize:[NSFont systemFontSize]]];
		[_clipButton setTarget:self];
		[_clipButton setDelegate:self];
	}
	
	return self;
}

- (void)dealloc
{
	// Release
	[_buttons release], _buttons = nil;
	_selectedButton = nil;
	[_bottomLineColor release], _bottomLineColor = nil;
	[_clipButton release], _clipButton = nil;
	
	[super dealloc];
}

//--------------------------------------------------------------//
#pragma mark	Selection
//--------------------------------------------------------------//

- (HMInfoBarButton*)selectedButton
{
	return _selectedButton;
}

- (void)selectButton:(HMInfoBarButton*)button
{
	// Filter
	if (!button || ![_buttons containsObject:button]) {
		return;
	}
	
	// Deselect
	[[self selectedButton] setState:NSOffState];
	
	// Select
	if ([button state] != NSOnState) {
		[button setState:NSOnState];
	}
	_selectedButton = button;
}

- (void)selectButtonAtIndex:(unsigned)index
{
	[self selectButton:[self buttonAtIndex:index]];
}

- (void)selectButtonWithTag:(int)tag
{
	[self selectButton:[self buttonWithTag:tag]];
}

//--------------------------------------------------------------//
#pragma mark	Finding
//--------------------------------------------------------------//

- (NSArray*)buttons
{
	return _buttons;
}

- (NSArray*)visibleButtons
{
	// Empty
	if (_clipButtonIndex <= 0 || _clipButtonIndex > [_buttons count]) {
		return [NSArray array];
	}
	
	// Return visible buttons
	NSRange range;
	range = NSMakeRange(0, _clipButtonIndex);
	
	return [_buttons subarrayWithRange:range];
}

- (NSArray*)clippedButtons
{
	int count;
	
	// Empty
	if (_clipButtonIndex < 0 || (count = [_buttons count] - _clipButtonIndex) < 1) {
		return [NSArray array];
	}
	
	// Return clipped buttons
	NSRange range;
	range = NSMakeRange(_clipButtonIndex, count);
	
	return [_buttons subarrayWithRange:range];
}

- (HMInfoBarButton*)buttonAtIndex:(unsigned)index
{
	if (index < 0 || index >= [_buttons count]) {
		return nil;
	}
	
	return [_buttons objectAtIndex:index];
}

- (HMInfoBarButton*)buttonWithTag:(int)tag
{
	NSEnumerator *enumerator;
	HMInfoBarButton *button;
	enumerator = [_buttons objectEnumerator];
	while (button = [enumerator nextObject]) {
		if ([button tag] == tag) {
			return button;
		}
	}
	
	return nil;
}

//--------------------------------------------------------------//
#pragma mark	Laying out the HMInfoBar
//--------------------------------------------------------------//

- (void)setButtons:(NSArray*)buttons
{
	// Remove all
	[self removeAllButtons];
	
	// Add buttons
	[self addButtons:buttons];
	
	// now _selectedButton is nil;
	// if buttons was nil, now, _buttons is empty array
}

- (void)addButton:(HMInfoBarButton*)button
{
	// Filter
	if (!button) {
		return;
	}
	
	// Add
	[_buttons addObject:button];
	
	// Add to view
	[self addSubview:button];
}

- (void)addButtons:(NSArray*)buttons
{
	// Filter
	if (!buttons) {
		return;
	}
	
	// Add
	[_buttons addObjectsFromArray:buttons];
	
	// Add to view
	NSEnumerator *enumerator;
	HMInfoBarButton *button;
	enumerator = [buttons objectEnumerator];
	while (button = [enumerator nextObject]) {
		[self addSubview:button];
	}
}

- (void)insertButton:(HMInfoBarButton*)button atIndex:(int)index
{
	// Filter
	if (!button || index < 0 || index >= [_buttons count]) {
		return;
	}
	
	// Insert
	[_buttons insertObject:button atIndex:index];
	
	// Add to view
	[self addSubview:button];
}

- (void)removeButton:(HMInfoBarButton*)button
{
	// Filter
	if (![_buttons containsObject:button]) {
		return;
	}
	
	// Update _selectedButton
	if (_selectedButton == button) {
		_selectedButton = nil;
	}
	
	// Remove
	[button removeFromSuperviewWithoutNeedingDisplay];
	[_buttons removeObjectIdenticalTo:button];
}

- (void)removeAllButtons
{
	// Update _selectedButton
	_selectedButton = nil;
	
	// Remove all
	[_buttons makeObjectsPerformSelector:@selector(removeFromSuperviewWithoutNeedingDisplay)];
	[_buttons removeAllObjects];
}

- (void)updateButtonsLayout
{
	[_clipButton removeFromSuperviewWithoutNeedingDisplay];
	
	// Filter
	if (![_buttons count]) {
		_clipButtonIndex = 0;
		[self setNeedsDisplay:YES];
		return;
	}
	
	// Hide all buttons
	[self _setButtons:_buttons toHidden:YES];
	
	// Get clip button space width
	float clipWidth;
	clipWidth = [_clipButton bounds].size.width + HMInfoBarClipButtonMarginX * 2;
	
	// Get margin x
	float marginX;
	marginX = [self marginX];
	
	// Layout ...
	NSRect spaceFrame;
	NSMutableArray *visibleButtons;
	HMInfoBarButton *button;
	spaceFrame = [self bounds];
	spaceFrame.size.width -= clipWidth;
	visibleButtons = [NSMutableArray array];
	unsigned i;
	
	// Layout buttons
	for (i = 0; i < [_buttons count]; i++) {
		NSSize maxSize;
		button = [_buttons objectAtIndex:i];
		maxSize = [button maxSize];
		if (i == [_buttons count] - 1) {		// Last button
			spaceFrame.size.width += clipWidth;
		}
		
		if (spaceFrame.size.width >= maxSize.width + marginX * 2) {
			// Set size to max
			[button setFrameSize:maxSize];
		}
		else if (spaceFrame.size.width >= [button minSize].width + marginX * 2) {
			// Set size to reduced
			[button setFrameSize:NSMakeSize(
					spaceFrame.size.width - marginX * 2,
					[button bounds].size.height)];
		}
		else {
			// Can't layout anymore
			break;
		}
		// Layout button & get remained space
		[self _layoutButton:button inSpace:&spaceFrame];
		[visibleButtons addObject:button];
	}
	_clipButtonIndex = [visibleButtons count];
	
	// Set visible buttons to visible
	[self _setButtons:visibleButtons toHidden:NO];
	
	// Show clipButton
	if ([visibleButtons count] < [_buttons count] && clipWidth <= [self bounds].size.width) {
		[self _layoutClipButton];
		[self addSubview:_clipButton];
	}
	
	// Update UI
	[self setNeedsDisplay:YES];
}

//--------------------------------------------------------------//
#pragma mark	Appearance
//--------------------------------------------------------------//

- (NSColor*)bottomLineColor
{
	return _bottomLineColor;
}

- (void)setBottomLineColor:(NSColor*)color
{
	// Filter
	if (!color) {
		return;
	}
	
	// Set
	[color retain];
	[_bottomLineColor release];
	_bottomLineColor = color;
	
	// Update UI
	[self setNeedsDisplay:YES];
}

- (float)marginX
{
	return _marginX;
}

- (void)setMarginX:(float)marginX
{
	_marginX = marginX;
}

- (void)resizeSubviewsWithOldSize:(NSSize)oldBoundsSize
{
	[self updateButtonsLayout];
}

- (BOOL)isFlipped
{
	return NO;
}

- (BOOL)isOpaque
{
	return YES;
}

- (void)drawRect:(NSRect)rect
{
	NSRect bounds;
	bounds = [self bounds];
	
	// Use cartesian coordinate system
	if ([self isFlipped]) {
		[[NSAffineTransform verticalFlipTransformForRect:bounds] concat];
	}
	
	// Draw 1px bottom line
	[[self bottomLineColor] set];
	NSRectFill(NSMakeRect(0, 0, bounds.size.width, 1));
	
	// Draw background image
	static NSImage *_backgroundImage = nil;
	NSRect bgRect;
	if (!_backgroundImage) {
        NSBundle*   bundle;
        bundle = [NSBundle bundleForClass:[self class]];
		_backgroundImage = [[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"infoBarGradient"]];
	}
	bgRect = bounds;
	bgRect.origin.y += 1;
	bgRect.size.height -= 1;
	[_backgroundImage drawInRect:bgRect
			fromRect:HMMakeRect(NSZeroPoint, [_backgroundImage size])
			operation:NSCompositeSourceOver
			fraction:1.0];
}

//--------------------------------------------------------------//
#pragma mark	HMMenuButton
//--------------------------------------------------------------//

- (NSMenu*)menuButton:(HMMenuButton*)menuButton menuForEvent:(NSEvent*)event
{
	// Get invisible buttons
	NSArray *clippedButtons;
	clippedButtons = [self clippedButtons];
	if (![clippedButtons count]) {
		return nil;
	}
	
	// Make menu
	NSMenu *menu;
	NSEnumerator *enumerator;
	HMInfoBarButton *button;
	menu = [[[NSMenu alloc] initWithTitle:@"InfoBarMenu"] autorelease];
	[menu setAutoenablesItems:NO];
	enumerator = [clippedButtons objectEnumerator];
	while (button = [enumerator nextObject]) {
		id <NSMenuItem> menuItem;
		menuItem = [menu addItemWithTitle:[button title]
				action:@selector(_menuItemAction:)
				keyEquivalent:@""];
		[menuItem setRepresentedObject:button];
		[menuItem setState:[button state]];
		[menuItem setEnabled:[button isEnabled]];
	}
	
	return menu;
}

- (void)_menuItemAction:(id)sender
{
	// Perform click
	HMInfoBarButton *button;
	button = [sender representedObject];
	[[sender representedObject] performClick:button];
}

//--------------------------------------------------------------//
#pragma mark	Util
//--------------------------------------------------------------//

- (void)_setButtons:(NSArray*)buttons toHidden:(BOOL)flag
{
	NSEnumerator *enumerator;
	HMInfoBarButton *button;
	enumerator = [buttons objectEnumerator];
	while (button = [enumerator nextObject]) {
		[button setHidden:flag];
	}
}

- (void)_layoutButton:(HMInfoBarButton*)button inSpace:(NSRect*)spaceFrame
{
	float marginX;
	marginX = [self marginX];
	
	// Set button frame
	NSRect buttonFrame;
	buttonFrame = [button frame];
	buttonFrame.origin.x = rintf(spaceFrame->origin.x + marginX);
	buttonFrame.origin.y = rintf((spaceFrame->size.height - buttonFrame.size.height) / 2.0);
	[button setFrame:buttonFrame];
	
	// Set space frame
	spaceFrame->origin.x = NSMaxX(buttonFrame);
	spaceFrame->size.width -= marginX + buttonFrame.size.width;
}

- (void)_layoutClipButton
{
	// Get bar bounds
	NSRect barBounds;
	barBounds = [self bounds];
	
	// Set clipButton frame
	NSRect frame;
	frame = [_clipButton frame];
	frame.origin.x = barBounds.size.width - HMInfoBarClipButtonMarginX - frame.size.width;
	frame.origin.y = rintf((barBounds.size.height - frame.size.height) / 2.0);
	[_clipButton setFrame:frame];
}

@end
