/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include "qfle3f.h"
#include "qfle3f_version.h"

vmk_uint64 qfle3fGetFcNodeName(void *clientData);
vmk_uint64 qfle3fGetFcPortName (void *clientData);
vmk_uint32 qfle3fGetFcPortId (void *clientData);
vmk_FcLinkSpeed qfle3fGetFcLinkSpeed (void *clientData);
vmk_FcPortType  qfle3fGetFcPortType(void *clientData);
vmk_FcPortState qfle3fGetFcPortState (void *clientData);
static VMK_ReturnStatus qfle3fGetFcTargetAttributes (void *pSCSI_Adapter,
						vmk_FcTargetAttrs *fcAttrib,
						vmk_uint32 channelNo,
						vmk_uint32 targetNo);
vmk_AdapterStatus qfle3fGetFcAdapterStatus (void *clientData);
static vmk_RescanLinkStatus qfle3fRescanFcLink (void *clientData);
static VMK_ReturnStatus
qfle3fGetFcAdapterAttributes (void *clientData,
				vmk_FcAdapterAttributes *adapterAttrib);
static VMK_ReturnStatus
qfle3fGetFcPortAttributes (void *clientData,
			vmk_uint32 portId,
			vmk_FcPortAttributes *portAttrib);
static VMK_ReturnStatus
qfle3fGetFcPortStatistics (void *clientData,
			vmk_FcPortStatistics *portStats);

/* Host attributes. */

vmk_uint32
qfle3fGetFcPortId (void *clientData)
{
    qfle3fHBA_t *hba = clientData;

    return hba->initiator_targetPortID;
}

vmk_FcLinkSpeed
qfle3fGetFcLinkSpeed(void *clientData)
{
	qfle3fHBA_t *hba = clientData;

/** Uplink auto negotiated speed */
    switch(hba->cnic->uplinkLinkState.speed) {
        case VMK_LINK_SPEED_AUTO:
        case VMK_LINK_SPEED_10_MBPS:
        case VMK_LINK_SPEED_100_MBPS:
        case VMK_LINK_SPEED_1000_MBPS:
        case VMK_LINK_SPEED_2500_MBPS:
        case VMK_LINK_SPEED_5000_MBPS:      return VMK_FC_SPEED_UNKNOWN;
        case VMK_LINK_SPEED_10000_MBPS:     return VMK_FC_SPEED_10GBIT;
        case VMK_LINK_SPEED_20000_MBPS:
        case VMK_LINK_SPEED_25000_MBPS:     return VMK_FC_SPEED_UNKNOWN;
        case VMK_LINK_SPEED_40000_MBPS:     return VMK_FC_SPEED_40GBIT;
        case VMK_LINK_SPEED_50000_MBPS:
        case VMK_LINK_SPEED_56000_MBPS:
        case VMK_LINK_SPEED_100000_MBPS:    return VMK_FC_SPEED_UNKNOWN;
    }
    return VMK_FC_PORTTYPE_UNKNOWN;
}

vmk_FcPortType
qfle3fGetFcPortType(void *clientData)
{
    return VMK_FC_PORTTYPE_NLPORT;
}

static VMK_ReturnStatus
qfle3fGetFcPortStatistics (void *clientData, vmk_FcPortStatistics *portStats)
{
	qfle3fHBA_t *hba = clientData;
	struct fcoe_statistics_params *fw_stats;
	int link_down = 0;

	fw_stats = (struct fcoe_statistics_params *)hba->statsBuffer;
	if (!fw_stats) {
		link_down = 1;
	} else {
		qfle3f_get_host_stats(hba);
	}

	portStats->secondsSinceLastReset = 0;
	portStats->errorFrames = 0; /* TODO: Do not keep count */
	portStats->linkFailureCount = 0;
	portStats->lossOfSyncCount = 0;
	portStats->lossOfSignalCount = 0;
	portStats->primitiveSeqProtocolErrCount = 0;
	portStats->invalidTxWordCount = 0;
	portStats->inputRequests = 0;
	portStats->outputRequests = 0;
	portStats->controlRequests = 0;
	portStats->inputMegabytes >>= 20;
	portStats->outputMegabytes >>= 20;
	portStats->dumpedFrames = 0;
	portStats->nosCount = 0;
	portStats->lipCount = 0;
	if (!link_down) {
		portStats->invalidCrcCount = fw_stats->rx_stat2.fc_crc_cnt;
		portStats->txFrames = fw_stats->tx_stat.fcoe_tx_pkt_cnt;
		portStats->txWords = (fw_stats->tx_stat.fcoe_tx_byte_cnt) / 4;
		portStats->rxFrames = fw_stats->rx_stat0.fcoe_rx_pkt_cnt;
		portStats->rxWords = (fw_stats->rx_stat0.fcoe_rx_byte_cnt) / 4;
	} else {
	        portStats->invalidCrcCount = 0;
	        portStats->txFrames = 0;
	        portStats->txWords = 0;
	        portStats->rxFrames = 0;
	        portStats->rxWords = 0;
	}

	return VMK_OK;
}

static vmk_uint64
qfle3fGetHostFabricName(void *clientData)
{
	qfle3fHBA_t *hba = clientData;
	vmk_uint64 nodeName;

    nodeName = qfle3f_get_wwnn(hba);

	return nodeName;
}

vmk_FcPortState
qfle3fGetFcPortState(void *clientData)
{
	qfle3fHBA_t *hba = clientData;
	int i;

	for (i = 0; (i < MAX_NUM_FABRICS + MAX_NUM_NPIV_LOGIN); i++) {
		struct ql_fcoe_fabric *Fabric = hba->qlfcoeAdapter->FabricArray[i];
		if (Fabric && Fabric->Flags & FABRIC_FLAGS_LOGGED_IN)
				return VMK_FC_PORTSTATE_ONLINE;
	}

	return VMK_FC_PORTSTATE_LINKDOWN;
}

vmk_uint64 qfle3fGetFcNodeName(void *clientData)
{
	qfle3fHBA_t *hba = clientData;

    return qfle3f_get_wwnn(hba);
}

vmk_uint64 qfle3fGetFcPortName (void *clientData)
{
	qfle3fHBA_t *hba = clientData;

    return qfle3f_get_wwpn(hba);
}

static VMK_ReturnStatus qfle3fGetFcPortAttributes (void *clientData,
						vmk_uint32 portId,
						vmk_FcPortAttributes *portAttrib)
{
	qfle3fHBA_t *hba = clientData;
	vmk_FcLinkSpeedBit speed = VMK_FC_SPEED_BIT_UNKNOWN;


	portAttrib->nodeWWN = qfle3fGetFcNodeName(clientData);
	portAttrib->portWWN = qfle3fGetFcPortName(clientData);
	portAttrib->portFcId = qfle3fGetFcPortId(clientData);
	portAttrib->portType = qfle3fGetFcPortType(clientData);
	portAttrib->portState = qfle3fGetFcPortState(clientData);

	//qfle3fGetSymPortName(hba, portAttrib->portSymbolicName);

    speed = VMK_FC_SPEED_BIT_10GBIT | VMK_FC_SPEED_BIT_1GBIT;

	portAttrib->portSupportedSpeed = speed;

	portAttrib->portSpeed = qfle3fGetFcLinkSpeed(clientData);
	/* portAttrib->portMaxFrameSize = (vmk_uint32) icb24->framePayloadSize; */
	portAttrib->fabricName = qfle3fGetHostFabricName(clientData);
	portAttrib->portSupportedClassOfService = BNX_FC_CLASS_3;
	/* TODO: Find out */
	/* portAttrib->portSupportedFc4Types[32];
	portAttrib->portActiveFc4Types[32];
	*/

	vmk_StringCopy(portAttrib->osDeviceName,
		vmk_ScsiGetAdapterName(hba->scsiAdapter), 256);

	portAttrib->numberOfDiscoveredPorts = hba->num_ofld_sess;

	return VMK_OK;
}

static VMK_ReturnStatus qfle3fGetFcTargetAttributes(void *clientData,
	vmk_FcTargetAttrs *fcAttrib, vmk_uint32 channelNo,
	vmk_uint32 targetNo)
{
	struct qfle3f_rport *rport = NULL;
	qfle3fHBA_t *hba = clientData;
	VMK_ReturnStatus status = VMK_OK;

	rport = qfle3f_findTargetByTargetID(hba, targetNo, VMK_FALSE);

	if (!rport)
		return VMK_FAILURE;

	fcAttrib->nodeName = rport->nodeName;
	fcAttrib->portName = rport->portName;
	fcAttrib->portId = hba->initiator_targetPortID;

	return status;

}

vmk_AdapterStatus qfle3fGetFcAdapterStatus(void *clientData)
{

	qfle3fHBA_t *hba = clientData;

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP))
		return VMK_ADAPTER_STATUS_ONLINE;
	else if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_GOING_DOWN) ||
            (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_LINK_DOWN)))
		return VMK_ADAPTER_STATUS_OFFLINE;
	else
		return VMK_ADAPTER_STATUS_UNKNOWN;

    return VMK_ADAPTER_STATUS_UNKNOWN;
}

static vmk_RescanLinkStatus qfle3fRescanFcLink(void *clientData)
{
	qfle3fHBA_t *hba = clientData;

	qfle3f_log(hba, LOG_INIT, "Issuing a Rescan FC Link");

	/* TODO: Need to perform rescan FC link */
	return VMK_RESCAN_LINK_SUCCEEDED;
}


static VMK_ReturnStatus qfle3fGetFcAdapterAttributes(void *clientData,
					vmk_FcAdapterAttributes *adapterAttrib)
{
	qfle3fHBA_t *hba = clientData;

	vmk_StringCopy(adapterAttrib->manufacturer, "QLOGIC", 19);

    vmk_StringFormat(adapterAttrib->model, sizeof(adapterAttrib->model),
                NULL, "BCM%s", hba->sym_name);
	vmk_StringFormat(adapterAttrib->modelDescription,
                sizeof(adapterAttrib->modelDescription),
                NULL,
                "Broadcom NetXtreme II %s FCoE Driver",
                hba->sym_name);

    adapterAttrib->nodeWWN = qfle3f_get_wwnn(hba);
    vmk_StringFormat(adapterAttrib->nodeSymbolicName,
                sizeof(adapterAttrib->nodeSymbolicName), NULL,
                "BCM%s", hba->sym_name);

	//qfle3fGetVpdField(hba, "MN", adapterAttrib->hardwareVersion, VMK_PAGE_SIZE);

	/* Driver Version */
	vmk_StringCopy(adapterAttrib->driverVersion, QFLE3F_VERSION, 32);

	/* Number of ports */
	adapterAttrib->numberOfPorts = 1;

	/* Driver Name */
	vmk_StringCopy(adapterAttrib->driverName, QFLE3F_DRIVER_NAME, 32);

	return VMK_OK;

}

static VMK_ReturnStatus qfle3fIssueFcPortReset(void *clientData, vmk_uint32 portId)
{
    qfle3fHBA_t *hba = clientData;
    VMK_ReturnStatus status = VMK_OK;

    qfle3f_log(hba, LOG_INIT, "Issuing a Port Reset");

    /* TODO: Fix this */
    return status;
}

static VMK_ReturnStatus qfle3f_issueFcoePortReset(void *clientData, vmk_uint32 portId) {
    qfle3fHBA_t *hba = clientData;

    qfle3f_log(hba, LOG_INIT, "Issuing a FCOE Port Reset");

    return VMK_OK;
}

static VMK_ReturnStatus qfle3f_getFcoePortStatistics(void *clientData, vmk_FcoePortStatistics *portStats)
{
	qfle3fHBA_t *hba = (qfle3fHBA_t *)clientData;
	struct fcoe_statistics_params *fw_stats;
	VMK_ReturnStatus status;
	int link_down = 0;

	qfle3f_log(hba, LOG_INIT, "Inside");

	fw_stats = (struct fcoe_statistics_params *)hba->statsBuffer;
	if (!fw_stats) {
		qfle3f_log(hba, LOG_APP, "No stats buffer allocated in driver");
		link_down = 1;
		goto hba_link_down;
	}

	if (qfle3f_sendStatRequest(hba)) {
		qfle3f_warning(hba, "failed to read stats info \n");
		link_down = 1;
		goto hba_link_down;
	}

	status = qfle3f_sleepWithCondition(hba, hba, (vmk_WorldEventID)&(hba->statRequestDone),
		VMK_LOCK_INVALID, 2, "Waiting from stats RAMROD",
		checkStatReqComplFlag);
	if (status != VMK_OK) {
		qfle3f_warning(hba, "failed to read stats info \n");
		link_down = 1;
	}

hba_link_down:
	if (!link_down) {
		portStats->txFrames = fw_stats->tx_stat.fcoe_tx_pkt_cnt;
		portStats->rxFrames = fw_stats->rx_stat0.fcoe_rx_pkt_cnt;
		portStats->txWords = fw_stats->tx_stat.fcoe_tx_byte_cnt;
		portStats->rxWords = fw_stats->rx_stat0.fcoe_rx_byte_cnt;
		portStats->errorFrames = fw_stats->rx_stat2.miss_frame_cnt;
		portStats->dumpedFrames = fw_stats->rx_stat2.drop_seq_cnt;
		portStats->lossOfSignalCount = fw_stats->rx_stat2.eofa_del_cnt;
	} else {
                portStats->txFrames = 0;
                portStats->rxFrames = 0;
                portStats->txWords = 0;
                portStats->rxWords = 0;
                portStats->errorFrames = 0;
                portStats->dumpedFrames = 0;
                portStats->lossOfSignalCount = 0;
	}
	portStats->secondsSinceLastReset = 0;
	portStats->invalidTxWordCount = 0;
	portStats->invalidCrcCount = 0;
	portStats->vlinkFailureCount = 0;
	portStats->missDiscAdvCount = 0;
	portStats->linkFailureCount = 0;

	/* protocol(fc4) statistics */
	portStats->inputRequests = 0;
	portStats->outputRequests = 0;
	portStats->controlRequests = 0;
	portStats->inputMegabytes = 0;
	portStats->outputMegabytes = 0;

	return VMK_OK;
}

static VMK_ReturnStatus qfle3f_getFCoEAdapterAttributes(void *clientData, vmk_FcoeAdapterAttrs *adapterAttrib)
{
    qfle3fHBA_t *hba = (qfle3fHBA_t *)clientData;

    /** \brief FCoE Controller MAC Address */
    vmk_Memcpy(adapterAttrib->fcoeContlrMacAddr, hba->dataSourceMacAddress, VMK_MAX_ETHERNET_MAC_LENGTH);
    /** \brief FCF MAC Address */
    vmk_Memcpy(adapterAttrib->fcfMacAddr, hba->fcf_mac, VMK_MAX_ETHERNET_MAC_LENGTH);
    /** \brief VN Port MAC Address*/
    vmk_Memcpy(adapterAttrib->vnPortMacAddr, hba->fpma, VMK_MAX_ETHERNET_MAC_LENGTH);
    /** \brief FIP Type */
    adapterAttrib->fip = VMK_FIP_TYPE_FPMA;
    /** \brief VLAN ID */
    qfle3f_log(hba, LOG_INFO, "vlanID = %d", hba->vlanID);
    adapterAttrib->vlanId = hba->vlanID;
    /** \brief Bandwidth Percentage */
    adapterAttrib->bwPercentage = 50;
    /** \brief FCoE MTU */
    adapterAttrib->frameSize = 0x800;
    /** \brief Underlying vmnic Name in case of CNA */
    vmk_Memcpy(adapterAttrib->vmnicName, vmk_NameToString(&hba->vmnicName), VMK_DEVICE_NAME_MAX_LENGTH);

    return VMK_OK;
}

void qfle3fInitFcoeAdapter(vmk_FcoeAdapter *qlaFcoeAdapter)
{
    qlaFcoeAdapter->getFCoEAdapterAttributes = qfle3f_getFCoEAdapterAttributes;
    qlaFcoeAdapter->getFcoePortStatistics = qfle3f_getFcoePortStatistics;
    qlaFcoeAdapter->issueFcoePortReset = qfle3f_issueFcoePortReset;


	qlaFcoeAdapter->fc.getFcNodeName = qfle3fGetFcNodeName;
	qlaFcoeAdapter->fc.getFcPortName = qfle3fGetFcPortName;
	qlaFcoeAdapter->fc.getFcPortId = qfle3fGetFcPortId;
	qlaFcoeAdapter->fc.getFcLinkSpeed = qfle3fGetFcLinkSpeed;
	qlaFcoeAdapter->fc.getFcPortType = qfle3fGetFcPortType;
	qlaFcoeAdapter->fc.getFcPortState = qfle3fGetFcPortState;
	qlaFcoeAdapter->fc.getFcTargetAttributes = qfle3fGetFcTargetAttributes;
	qlaFcoeAdapter->fc.linkTimeout = 30;
	qlaFcoeAdapter->fc.ioTimeout = 20;   /* TODO: Check it */
	qlaFcoeAdapter->fc.getFcAdapterStatus = qfle3fGetFcAdapterStatus;
	qlaFcoeAdapter->fc.rescanFcLink = qfle3fRescanFcLink;
	qlaFcoeAdapter->fc.getFcAdapterAttributes = qfle3fGetFcAdapterAttributes;
	qlaFcoeAdapter->fc.getFcPortAttributes = qfle3fGetFcPortAttributes;
	qlaFcoeAdapter->fc.getFcPortStatistics = qfle3fGetFcPortStatistics;

	qlaFcoeAdapter->fc.issueFcPortReset  = qfle3fIssueFcPortReset;
}
