//**************************************************************************************************
//                                          Config.cpp                                             *
//                                         ------------                                            *
// Started     : 2016-09-25                                                                        *
// Last Update : 2020-11-01                                                                        *
// Copyright   : (C) 2016-2020 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "Config.hpp"

//**************************************************************************************************
// Allocate storage for static data members.

wxConfig * Config::m_poCfg = NULL;

//**************************************************************************************************
// Constructor.

Config::Config( void )
{
}

//**************************************************************************************************
// Destructor.

Config::~Config( )
{
}

//**************************************************************************************************
// Open a configuration file.
//
// Argument List :
//   rosFileName - A string containing the configuration file name, if empty use m_ofnFileName
//
// Return Values :
//   Success - true
//   Failure - false

bool  Config::bOpen( const wxString & rosFileName )
{
  wxConfig  * poCfg;
  wxFileName  ofn1;

  // A configuration file must be open
  if( bIsOpen( ) )                      return( false );

  // Set the RC file name
  if( ! rosFileName.IsEmpty( ) )
    if( ! bSetFileName( rosFileName ) ) return( false );

  // Configure and create the global configuration object
  m_poCfg = new wxConfig( wxEmptyString, wxEmptyString, m_ofnFileName.GetFullPath( ) );
  poCfg = (wxConfig *) wxConfig::Set( m_poCfg );
  if( poCfg != NULL ) delete poCfg;  // Delete the old wxConfig object if one exists

  // Record a default to the config file whenever attempting to read a value which is not present
  m_poCfg->SetRecordDefaults( true );

  return( true );
}

//**************************************************************************************************
// Close the configuration file.
//
// Return Values:
//   Success - true
//   Failure - false

bool  Config::bClose( void )
{
  // A configuration file must be open
  if( ! bIsOpen( ) ) return( true );

  // Delete the global configuration object
  delete wxConfig::Set( (wxConfig *) NULL );
  m_poCfg = NULL;

  return( true );
}

//**************************************************************************************************
// Permanently writes all changes to the configuration on disk.
//
// Return Values:
//   Success - true
//   Failure - false

bool  Config::bFlush( void )
{
  // A configuration file must be open
  if( ! bIsOpen( ) ) return( false );

  m_poCfg->Flush( );

  return( true );
}

//**************************************************************************************************
// Clean / rebuild the configuration file.
//
// Over time configuration options change, new ones are introduced, configuration groups change,
// names changes and old configuration options are dropped. Consequently the configuration file can
// become cluttered with outdated information. This function cleans out all this out-moded data.
//
// Return Values:
//   Success - true
//   Failure - false

bool  Config::bClean( void )
{
  bool  bRtn=true;

  // A configuration file must be open
  if( ! bIsOpen( ) ) return( false );

  // Create temporary storage for all configuration parameters
  wxString         osAppVersion;    // MAIN           group
  uint             uiFrmLayout;
  eTypeTmpFileMgt  eTmpFileMgt;
  uint             uiPrecision;
  uint             uiNbkMaxLns;
  uint             uiSpnPeriod;
  uint             uiToolTipDly;
  bool             bToolTips;
  bool             bAutoCfgClean;
  bool             bKeepNetLst;
  int              iMainPosnX;
  int              iMainPosnY;
  int              iMainSizeW;
  int              iMainSizeH;
  int              iHelpPosnX;      // HELP           group
  int              iHelpPosnY;
  int              iHelpSizeW;
  int              iHelpSizeH;
  wxString         osNetLstFile;    // FILES          group
  wxString         osSchemFiles;
  wxString         osRecentFiles;
  wxString         osDirLastAcc;
  wxString         osGuileProc;     // NETLISTER      group
  bool             bVerboseMode;
  bool             bIncludeMode;
  bool             bEmbedMode;
  bool             bNoMungeMode;
  eTypeSimEng      eSimEngine;      // SIM_ENGINE     group
  eTypeCmd         eAnalysis;
  eTypeUnits       ePhaseUnits;
  bool             bSyncSigSrcs;
  bool             bSyncTemps;
  bool             bAutoRegen;
  eTypeEDA         eEdaToolSuite;   // EDA_TOOL_SUITE group
  eTypeDatVwr      eDataViewer;     // DATA_VIEWER    group
  wxString         osCalculator;    // CALCULATOR     group
  wxString         osPdfViewer;     // PDF_VIEWER     group

  // Get all configuration parameter values
  osAppVersion  = APP_VERSION;            // MAIN           group
  uiFrmLayout   =  uiGetFrmLayout   ( );
  eTmpFileMgt   =   eGetTmpFileMgt  ( );
  uiPrecision   =  uiGetPrecision   ( );
  uiNbkMaxLns   =  uiGetNbkMaxLns   ( );
  uiSpnPeriod   =  uiGetSpnPeriod   ( );
  uiToolTipDly  =  uiGetToolTipDly  ( );
  bToolTips     =   bGetToolTips    ( );
  bAutoCfgClean =   bGetAutoCfgClean( );
  bKeepNetLst   =   bGetKeepNetLst  ( );
  iMainPosnX    =   iGetMainPosnX   ( );
  iMainPosnY    =   iGetMainPosnY   ( );
  iMainSizeW    =   iGetMainSizeW   ( );
  iMainSizeH    =   iGetMainSizeH   ( );
  iHelpPosnX    =   iGetHelpPosnX   ( );  // HELP           group
  iHelpPosnY    =   iGetHelpPosnY   ( );
  iHelpSizeW    =   iGetHelpSizeW   ( );
  iHelpSizeH    =   iGetHelpSizeH   ( );
  osNetLstFile  = rosGetNetLstFile  ( );  // FILES          group
  osSchemFiles  = rosGetSchemFiles  ( );
  osRecentFiles = rosGetRecentFiles ( );
  osDirLastAcc  = rosGetDirLastAcc  ( );
  osGuileProc   = rosGetGuileProc   ( );  // NETLISTER        group
  bVerboseMode  =   bGetVerboseMode ( );
  bIncludeMode  =   bGetIncludeMode ( );
  bEmbedMode    =   bGetEmbedMode   ( );
  bNoMungeMode  =   bGetNoMungeMode ( );
  eSimEngine    =   eGetSimEngine   ( );  // SIM_ENGINE     group
  eAnalysis     =   eGetAnalysis    ( );
  ePhaseUnits   =   eGetPhaseUnits  ( );
  bSyncSigSrcs  =   bGetSyncSigSrcs ( );
  bSyncTemps    =   bGetSyncTemps   ( );
  bAutoRegen    =   bGetAutoRegen   ( );
  eEdaToolSuite =   eGetEdaToolSuite( );  // EDA_TOOL_SUITE group
  eDataViewer   =   eGetDataViewer  ( );  // DATA_VIEWER    group
  osCalculator  = rosGetCalculator  ( );  // CALCULATOR     group
  osPdfViewer   = rosGetPdfViewer   ( );  // PDF_VIEWER     group

  // Delete the whole underlying object (disk file, registry key, ...)
  m_poCfg->DeleteAll( );

  // Close this configuration object
  if( ! bClose( ) )                        bRtn = false;

  // Re-open this configuration object
  if( ! bOpen( ) )                         bRtn = false;

  // Re-build the configuration object
  if( ! bSetAppVersion  ( osAppVersion  ) ) bRtn = false;  // MAIN           group
  if( ! bSetFrmLayout   ( uiFrmLayout   ) ) bRtn = false;
  if( ! bSetTmpFileMgt  ( eTmpFileMgt   ) ) bRtn = false;
  if( ! bSetPrecision   ( uiPrecision   ) ) bRtn = false;
  if( ! bSetNbkMaxLns   ( uiNbkMaxLns   ) ) bRtn = false;
  if( ! bSetSpnPeriod   ( uiSpnPeriod   ) ) bRtn = false;
  if( ! bSetToolTipDly  ( uiToolTipDly  ) ) bRtn = false;
  if( ! bSetToolTips    ( bToolTips     ) ) bRtn = false;
  if( ! bSetAutoCfgClean( bAutoCfgClean ) ) bRtn = false;
  if( ! bSetKeepNetLst  ( bKeepNetLst   ) ) bRtn = false;
  if( ! bSetMainPosnX   ( iMainPosnX    ) ) bRtn = false;
  if( ! bSetMainPosnY   ( iMainPosnY    ) ) bRtn = false;
  if( ! bSetMainSizeW   ( iMainSizeW    ) ) bRtn = false;
  if( ! bSetMainSizeH   ( iMainSizeH    ) ) bRtn = false;
  if( ! bSetHelpPosnX   ( iHelpPosnX    ) ) bRtn = false;  // HELP           group
  if( ! bSetHelpPosnY   ( iHelpPosnY    ) ) bRtn = false;
  if( ! bSetHelpSizeW   ( iHelpSizeW    ) ) bRtn = false;
  if( ! bSetHelpSizeH   ( iHelpSizeH    ) ) bRtn = false;
  if( ! bSetNetLstFile  ( osNetLstFile  ) ) bRtn = false;  // FILES          group
  if( ! bSetSchemFiles  ( osSchemFiles  ) ) bRtn = false;
  if( ! bSetRecentFiles ( osRecentFiles ) ) bRtn = false;
  if( ! bSetDirLastAcc  ( osDirLastAcc  ) ) bRtn = false;
  if( ! bSetGuileProc   ( osGuileProc   ) ) bRtn = false;  // NETLISTER      group
  if( ! bSetVerboseMode ( bVerboseMode  ) ) bRtn = false;
  if( ! bSetIncludeMode ( bIncludeMode  ) ) bRtn = false;
  if( ! bSetEmbedMode   ( bEmbedMode    ) ) bRtn = false;
  if( ! bSetNoMungeMode ( bNoMungeMode  ) ) bRtn = false;
  if( ! bSetSimEngine   ( eSimEngine    ) ) bRtn = false;  // SIM_ENGINE     group
  if( ! bSetAnalysis    ( eAnalysis     ) ) bRtn = false;
  if( ! bSetPhaseUnits  ( ePhaseUnits   ) ) bRtn = false;
  if( ! bSetSyncSigSrcs ( bSyncSigSrcs  ) ) bRtn = false;
  if( ! bSetSyncTemps   ( bSyncTemps    ) ) bRtn = false;
  if( ! bSetAutoRegen   ( bAutoRegen    ) ) bRtn = false;
  if( ! bSetEdaToolSuite( eEdaToolSuite ) ) bRtn = false;  // EDA_TOOL_SUITE group
  if( ! bSetDataViewer  ( eDataViewer   ) ) bRtn = false;  // DATA_VIEWER    group
  if( ! bSetCalculator  ( osCalculator  ) ) bRtn = false;  // CALCULATOR     group
  if( ! bSetPdfViewer   ( osPdfViewer   ) ) bRtn = false;  // PDF_VIEWER     group

  // Permanently writes all changes to the configuration on disk
  if( ! bFlush( ) )                        bRtn = false;

  return( bRtn );
}

//**************************************************************************************************
// Set the configuration file name.
//
// Note : if no path is specified the users home directory is assumed
//
// Argument List :
//   rosFileName - A string containing the configuration file name
//
// Return Values:
//   Success - true
//   Failure - false

bool  Config::bSetFileName( const wxString & rosFileName )
{
  wxFileName  ofn1;

  // The configuration object must be closed
  if( bIsOpen( ) )              return( false );

  // Check that the RC file name argument is valid (if no path is specified assume users home dir.)
  ofn1 = rosFileName;
#ifndef __WXMSW__
  if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxGetenv( "HOME" ) );
#else
  if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxGetenv( "USERPROFILE" ) );
#endif
  if( ! ofn1.IsDirWritable( ) ) return( false );

  // Check that the file name attribute it valid
  if( ! ofn1.IsOk( ) )          return( false );

  // Set the object attribute
  m_ofnFileName = ofn1;

  return( true );
}

//**************************************************************************************************
// Get the configuration file name as a string.
//
// Return Values:
//   The configuration file name or an empty string

const wxString & Config::rosGetFileName( void )
{
  static  wxString  osFileName;

  osFileName = m_ofnFileName.GetFullPath( );

  return( osFileName );
}

//**************************************************************************************************
// Generic set functions for writing a parameter into the configuration object.
//
// Argument List :
//   rosGroup - The group name
//   rosKey   - The key (parameter) name
//   *Value   - The value to set
//
// Return Values :
//   Success - true
//   Failure - false

bool  Config::bSet( const wxString & rosGroup, const wxString & rosKey, int iValue )
{
  // The configuration object must be closed
  if( ! bIsOpen( ) ) return( false );

  m_poCfg->SetPath( rosGroup );
  m_poCfg->Write( rosKey, iValue );

  return( true );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSet( const wxString & rosGroup, const wxString & rosKey, bool bValue )
{
  // The configuration object must be closed
  if( ! bIsOpen( ) ) return( false );

  m_poCfg->SetPath( rosGroup );
  m_poCfg->Write( rosKey, bValue );

  return( true );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSet( const wxString & rosGroup, const wxString & rosKey, const wxString & rosValue )
{
  // The configuration object must be closed
  if( ! bIsOpen( ) ) return( false );

  m_poCfg->SetPath( rosGroup );
  m_poCfg->Write( rosKey, rosValue );

  return( true );
}

//**************************************************************************************************
// Generic get functions for reading a parameter from the configuration object.
//
// Argument List :
//   rosGroup - The group name
//   rosKey   - The key (parameter) name
//   *Default - The default value
//
// Return Values :
//   Success - The configuration value
//   Failure - The default       value

int  Config::iGet( const wxString & rosGroup, const wxString & rosKey, int iDefault )
{
  long  li1;

  if( bIsOpen( ) )
  {
    m_poCfg->SetPath( rosGroup );
    m_poCfg->Read( rosKey, &li1, long( iDefault ) );
  }
  else li1 = long( iDefault );

  return( int( li1 ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGet( const wxString & rosGroup, const wxString & rosKey, bool bDefault )
{
  bool  b1;

  if( bIsOpen( ) )
  {
    m_poCfg->SetPath( rosGroup );
    m_poCfg->Read( rosKey, &b1, bDefault );
  }
  else b1 = bDefault;

  return( b1 );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGet( const wxString & rosGroup, const wxString & rosKey,
                                 const wxString & rosDefault )
{
  static  wxString  os1;

  if( bIsOpen( ) )
  {
    m_poCfg->SetPath( rosGroup );
    m_poCfg->Read( rosKey, &os1, rosDefault );
  }
  else os1 = rosDefault;

  return( os1 );
}

//**************************************************************************************************
// Set configuration parameter value functions.
//
// Argument List :
//   The configuration parameter value to be updated
//
// Return Values:
//   Success - true
//   Failure - false

bool  Config::bSetAppVersion( const wxString & rosAppVersion )
{
  return( bSet( "/MAIN", "AppVersion", rosAppVersion ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetFrmLayout( uint uiLayout )
{
  return( bSet( "/MAIN", "FrmLayout", int( uiLayout ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetTmpFileMgt( eTypeTmpFileMgt eTmpFileMgt )
{
  return( bSet( "/MAIN", "TmpFileMgt", int( eTmpFileMgt ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetPrecision( uint uiFltRes )
{
  return( bSet( "/MAIN", "Precision", int( uiFltRes ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetNbkMaxLns( uint uiLnCnt )
{
  return( bSet( "/MAIN", "NbkMaxLns", int( uiLnCnt ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSpnPeriod( uint ui_mSec )
{
  return( bSet( "/MAIN", "SpnPeriod", int( ui_mSec ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetToolTipDly( uint ui_mSec )
{
  return( bSet( "/MAIN", "ToolTipDly", int( ui_mSec ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetToolTips( bool bEnable )
{
  return( bSet( "/MAIN", "ToolTips", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetAutoCfgClean( bool bEnable )
{
  return( bSet( "/MAIN", "AutoCfgClean", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetKeepNetLst( bool bEnable )
{
  return( bSet( "/MAIN", "KeepNetLst", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetMainPosnX( int iPosnX )
{
  return( bSet( "/MAIN", "PosnX", iPosnX ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetMainPosnY( int iPosnY )
{
  return( bSet( "/MAIN", "PosnY", iPosnY ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetMainSizeW( int iSizeW )
{
  return( bSet( "/MAIN", "SizeW", iSizeW ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetMainSizeH( int iSizeH )
{
  return( bSet( "/MAIN", "SizeH", iSizeH ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetHelpPosnX( int iPosnX )
{
  return( bSet( "/HELP", "PosnX", iPosnX ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetHelpPosnY( int iPosnY )
{
  return( bSet( "/HELP", "PosnY", iPosnY ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetHelpSizeW( int iSizeW )
{
  return( bSet( "/HELP", "SizeW", iSizeW ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetHelpSizeH( int iSizeH )
{
  return( bSet( "/HELP", "SizeH", iSizeH ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetNetLstFile( const wxString & rosFile )
{
  return( bSet( "/FILES", "NetList", rosFile ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSchemFiles( const wxString & rosFiles )
{
  return( bSetSchemFiles( roasStrToArr( rosFiles ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSchemFiles( const wxArrayString & roasFiles )
{
  return( bSet( "/FILES", "Schematics", rosArrToStr( roasFiles ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetRecentFiles( const wxString & rosFiles )
{
  return( bSetRecentFiles( roasStrToArr( rosFiles ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetRecentFiles( const wxArrayString & roasFiles )
{
  return( bSet( "/FILES", "Recent", rosArrToStr( roasFiles ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetDirLastAcc( const wxString & rosFile )
{
  return( bSet( "/FILES", "DirLastAcc", rosFile ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetGuileProc( const wxString & rosGuileProc )
{
  return( bSet( "/NETLISTER", "GuileProc", rosGuileProc ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetVerboseMode( bool bEnable )
{
  return( bSet( "/NETLISTER", "VerboseMode", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetIncludeMode( bool bEnable )
{
  return( bSet( "/NETLISTER", "IncludeMode", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetEmbedMode( bool bEnable )
{
  return( bSet( "/NETLISTER", "EmbedMode", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetNoMungeMode( bool bEnable )
{
  return( bSet( "/NETLISTER", "NoMungeMode", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSimEngine( const wxString & rosSimEngine )
{
  // Check that the viewer name is valid
  if( eStrToEnumEng( rosSimEngine ) == eSIMR_NONE ) return( false );

  return( bSet( "/SIM_ENGINE", "Project", rosSimEngine ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSimEngine( eTypeSimEng eSimEngine )
{
  return( bSetSimEngine( rosEnumEngToStr( eSimEngine ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetAnalysis( eTypeCmd eAnalysis )
{
  return( bSet( "/SIM_ENGINE", "Analysis", rosEnumCmdToStr( eAnalysis ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetPhaseUnits( eTypeUnits eUnits )
{
  return( bSet( "/SIM_ENGINE", "PhaseUnits", rosEnumPhaToStr( eUnits ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSyncSigSrcs( bool bEnable )
{
  return( bSet( "/SIM_ENGINE", "SyncSigSrcs", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetSyncTemps( bool bEnable )
{
  return( bSet( "/SIM_ENGINE", "SyncTemps", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetAutoRegen( bool bEnable )
{
  return( bSet( "/SIM_ENGINE", "AutoRegen", bEnable ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetEdaToolSuite( const wxString & rosEdaToolSuite )
{
  return( bSet( "/EDA_TOOL_SUITE", "Project", rosEdaToolSuite ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetEdaToolSuite( eTypeEDA eEDA )
{
  return( bSetEdaToolSuite( rosEnumEdaToStr( eEDA ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetDataViewer( const wxString & rosDataViewer )
{
  // Check that the viewer name is valid
  if( eStrToEnumVwr( rosDataViewer ) == eDVWR_NONE ) return( false );

  return( bSet( "/DATA_VIEWER", "Binary", rosDataViewer ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetDataViewer( eTypeDatVwr eDataViewer )
{
  return( bSetDataViewer( rosEnumVwrToStr( eDataViewer ).Lower( ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetCalculator( const wxString & rosCalculator )
{
  return( bSet( "/CALCULATOR", "Binary", rosCalculator ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bSetPdfViewer( const wxString & rosPdfViewer )
{
  return( bSet( "/PDF_VIEWER", "Binary", rosPdfViewer ) );
}

//**************************************************************************************************
// Get configuration parameter value functions.
//
// Return Values:
//   Success - The value stored in the configuration file
//   Failure - The default value

const wxString & Config::rosGetAppVersion( void )
{
  return( rosGet( "/MAIN", "AppVersion", CFG_DEF_APP_VERSION ) );
}

//--------------------------------------------------------------------------------------------------

uint  Config::uiGetFrmLayout( void )
{
  return( iGet( "/MAIN", "FrmLayout", CFG_DEF_FRMLAYOUT ) );
}

//--------------------------------------------------------------------------------------------------

eTypeTmpFileMgt  Config::eGetTmpFileMgt( void )
{
  return( (eTypeTmpFileMgt) iGet( "/MAIN", "TmpFileMgt", CFG_DEF_TMPFILEMGT ) );
}

//--------------------------------------------------------------------------------------------------

uint  Config::uiGetPrecision( void )
{
  return( uint( iGet( "/MAIN", "Precision", CFG_DEF_PRECISION ) ) );
}

//--------------------------------------------------------------------------------------------------

uint  Config::uiGetNbkMaxLns( void )
{
  return( uint( iGet( "/MAIN", "NbkMaxLns", CFG_DEF_NBKMAXLNS ) ) );
}

//--------------------------------------------------------------------------------------------------

uint  Config::uiGetSpnPeriod( void )
{
  return( uint( iGet( "/MAIN", "SpnPeriod", CFG_DEF_SPNPERIOD ) ) );
}

//--------------------------------------------------------------------------------------------------

uint  Config::uiGetToolTipDly( void )
{
  return( uint( iGet( "/MAIN", "ToolTipDly", CFG_DEF_TOOLTIPDLY ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetToolTips( void )
{
  return( bGet( "/MAIN", "ToolTips", CFG_DEF_TOOLTIPS ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetAutoCfgClean( void )
{
  return( bGet( "/MAIN", "AutoCfgClean", CFG_DEF_AUTOCFGCLEAN ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetKeepNetLst( void )
{
  return( bGet( "/MAIN", "KeepNetLst", CFG_DEF_KEEPNETLST ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetMainPosnX( void )
{
  return( iGet( "/MAIN", "PosnX", CFG_DEF_MAINPOSNX ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetMainPosnY( void )
{
  return( iGet( "/MAIN", "PosnY", CFG_DEF_MAINPOSNY ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetMainSizeW( void )
{
  return( iGet( "/MAIN", "SizeW", CFG_DEF_MAINSIZEW ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetMainSizeH( void )
{
  return( iGet( "/MAIN", "SizeH", CFG_DEF_MAINSIZEH ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetHelpPosnX( void )
{
  return( iGet( "/HELP", "PosnX", CFG_DEF_HELPPOSNX ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetHelpPosnY( void )
{
  return( iGet( "/HELP", "PosnY", CFG_DEF_HELPPOSNY ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetHelpSizeW( void )
{
  return( iGet( "/HELP", "SizeW", CFG_DEF_HELPSIZEW ) );
}

//--------------------------------------------------------------------------------------------------

int  Config::iGetHelpSizeH( void )
{
  return( iGet( "/HELP", "SizeH", CFG_DEF_HELPSIZEH ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetNetLstFile( void )
{
  return( rosGet( "/FILES", "NetList", wxEmptyString ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetSchemFiles( void )
{
  return( rosArrToStr( roasGetSchemFiles( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxArrayString & Config::roasGetSchemFiles( void )
{
  return( roasStrToArr( rosGet( "/FILES", "Schematics", wxEmptyString ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetRecentFiles( void )
{
  return( rosArrToStr( roasGetRecentFiles( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxArrayString & Config::roasGetRecentFiles( void )
{
  return( roasStrToArr( rosGet( "/FILES", "Recent", wxEmptyString ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetDirLastAcc( void )
{
  return( rosGet( "/FILES", "DirLastAcc", wxGetHomeDir( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetGuileProc( void )
{
  return( rosGet( "/NETLISTER", "GuileProc", CFG_DEF_GUILEPROC ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetVerboseMode( void )
{
  return( bGet( "/NETLISTER", "VerboseMode", CFG_DEF_VERBOSEMODE ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetIncludeMode( void )
{
  return( bGet( "/NETLISTER", "IncludeMode", CFG_DEF_INCLUDEMODE ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetEmbedMode( void )
{
  return( bGet( "/NETLISTER", "EmbedMode", CFG_DEF_EMBEDMODE ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetNoMungeMode( void )
{
  return( bGet( "/NETLISTER", "NoMungeMode", CFG_DEF_NOMUNGEMODE ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetSimEngine( void )
{
  return( rosGet( "/SIM_ENGINE", "Project", CFG_DEF_SIMENGINE ) );
}

//--------------------------------------------------------------------------------------------------

eTypeSimEng  Config::eGetSimEngine( void )
{
  return( eStrToEnumEng( rosGetSimEngine( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetAnalysis( void )
{
  return( rosGet( "/SIM_ENGINE", "Analysis", CFG_DEF_ANALYSIS ) );
}

//--------------------------------------------------------------------------------------------------

eTypeCmd  Config::eGetAnalysis( void )
{
  return( eStrToEnumCmd( rosGetAnalysis( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetPhaseUnits( void )
{
  return( rosGet( "/SIM_ENGINE", "PhaseUnits", CFG_DEF_PHASEUNITS ) );
}

//--------------------------------------------------------------------------------------------------

eTypeUnits  Config::eGetPhaseUnits( void )
{
  return( eStrToEnumPha( rosGetPhaseUnits( ) ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetSyncSigSrcs( void )
{
  return( bGet( "/SIM_ENGINE", "SyncSigSrcs", CFG_DEF_SYNCSIGSRCS ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetSyncTemps( void )
{
  return( bGet( "/SIM_ENGINE", "SyncTemps", CFG_DEF_SYNCTEMPS ) );
}

//--------------------------------------------------------------------------------------------------

bool  Config::bGetAutoRegen( void )
{
  return( bGet( "/SIM_ENGINE", "AutoRegen", CFG_DEF_AUTOREGEN ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetEdaToolSuite( void )
{
  return( rosGet( "/EDA_TOOL_SUITE", "Project", CFG_DEF_EDATOOLSUITE ) );
}

//--------------------------------------------------------------------------------------------------

eTypeEDA  Config::eGetEdaToolSuite( void )
{
  return( eStrToEnumEda( rosGetEdaToolSuite( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetDataViewer( void )
{
  return( rosGet( "/DATA_VIEWER", "Binary", CFG_DEF_DATAVIEWER ) );
}

//--------------------------------------------------------------------------------------------------

eTypeDatVwr  Config::eGetDataViewer( void )
{
  return( eStrToEnumVwr( rosGetDataViewer( ) ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetCalculator( void )
{
  return( rosGet( "/CALCULATOR", "Binary", CFG_DEF_CALCULATOR ) );
}

//--------------------------------------------------------------------------------------------------

const wxString & Config::rosGetPdfViewer( void )
{
  return( rosGet( "/PDF_VIEWER", "Binary", CFG_DEF_PDFVIEWER ) );
}

//**************************************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  Config::Print( const wxString & rosPrefix )
{
  std::cout << rosPrefix.mb_str( ) << "rosGetFileName( )     : " << rosGetFileName( )    .mb_str( ) << '\n';
  std::cout << '\n';

  // MAIN group
  std::cout << rosPrefix.mb_str( ) << "rosGetAppVersion( )   : " << rosGetAppVersion( )  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << " uiGetFrmLayout( )    : " << uiGetFrmLayout( )         << '\n';
  std::cout << rosPrefix.mb_str( ) << "  eGetTmpFileMgt( )   : " << eGetTmpFileMgt( )         << '\n';
  std::cout << rosPrefix.mb_str( ) << " uiGetPrecision( )    : " << uiGetPrecision( )         << '\n';
  std::cout << rosPrefix.mb_str( ) << " uiGetNbkMaxLns( )    : " << uiGetNbkMaxLns( )         << '\n';
  std::cout << rosPrefix.mb_str( ) << " uiGetSpnPeriod( )    : " << uiGetSpnPeriod( )         << '\n';
  std::cout << rosPrefix.mb_str( ) << " uiGetToolTipDly( )   : " << uiGetToolTipDly( )        << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetToolTips( )     : " << (bGetToolTips( )     ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetAutoCfgClean( ) : " << (bGetAutoCfgClean( ) ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetKeepNetLst( )   : " << (bGetKeepNetLst( )   ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetMainPosnX( )    : " << iGetMainPosnX( )          << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetMainPosnY( )    : " << iGetMainPosnY( )          << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetMainSizeW( )    : " << iGetMainSizeW( )          << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetMainSizeH( )    : " << iGetMainSizeH( )          << '\n';
  std::cout << '\n';

  // HELP group
  std::cout << rosPrefix.mb_str( ) << "  iGetHelpPosnX( )    : " << iGetHelpPosnX( )          << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetHelpPosnY( )    : " << iGetHelpPosnY( )          << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetHelpSizeW( )    : " << iGetHelpSizeW( )          << '\n';
  std::cout << rosPrefix.mb_str( ) << "  iGetHelpSizeH( )    : " << iGetHelpSizeH( )          << '\n';
  std::cout << '\n';

  // FILES group
  std::cout << rosPrefix.mb_str( ) << "rosGetNetLstFile( )   : " << rosGetNetLstFile( )  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "rosGetSchemFiles( )   : " << rosGetSchemFiles( )  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "rosGetRecentFiles( )  : " << rosGetRecentFiles( ) .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "rosGetDirLastAcc( )   : " << rosGetDirLastAcc( )  .mb_str( ) << '\n';
  std::cout << '\n';

  // NETLISTER group
  std::cout << rosPrefix.mb_str( ) << "  bGetVerboseMode( )  : " << (bGetVerboseMode( )  ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "rosGetGuileProc( )    : " << rosGetGuileProc( )   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetIncludeMode( )  : " << (bGetIncludeMode( )  ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetEmbedMode( )    : " << (bGetEmbedMode( )    ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetNoMungeMode( )  : " << (bGetNoMungeMode( )  ?"true":"false") << '\n';
  std::cout << '\n';

  // SIM_ENGINE group
  std::cout << rosPrefix.mb_str( ) << "rosGetSimEngine( )    : " << rosGetSimEngine( )   .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "rosGetAnalysis( )     : " << rosGetAnalysis( )    .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "rosGetPhaseUnits( )   : " << rosGetPhaseUnits( )  .mb_str( ) << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetSyncSigSrcs( )  : " << (bGetSyncSigSrcs( )  ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetSyncTemps( )    : " << (bGetSyncTemps( )    ?"true":"false") << '\n';
  std::cout << rosPrefix.mb_str( ) << "  bGetAutoRegen( )    : " << (bGetAutoRegen( )    ?"true":"false") << '\n';
  std::cout << '\n';

  // EDA_TOOL_SUITE group
  std::cout << rosPrefix.mb_str( ) << "rosGetEdaToolSuite( ) : " << rosGetEdaToolSuite( ).mb_str( ) << '\n';

  // DATA_VIEWER group
  std::cout << rosPrefix.mb_str( ) << "rosGetDataViewer( )   : " << rosGetDataViewer( )  .mb_str( ) << '\n';

  // CALCULATOR group
  std::cout << rosPrefix.mb_str( ) << "rosGetCalculator( )   : " << rosGetCalculator( )  .mb_str( ) << '\n';

  // PDF_VIEWER group
  std::cout << rosPrefix.mb_str( ) << "rosGetPdfViewer( )    : " << rosGetPdfViewer( )   .mb_str( ) << '\n';
}

//**************************************************************************************************
//                                          Test Utility                                           *
//**************************************************************************************************

#ifdef TEST_CONFIG

using  namespace  std;

// wxWidgets Includes

#include <wx/cmdline.h>

// Function prototypes

void  Usage( char * psAppName );

// Declare a structure defining the command line syntax
static  const  wxCmdLineEntryDesc  tCmdLnDesc[] =
{
  { wxCMD_LINE_SWITCH, "h", "", "", wxCMD_LINE_VAL_NONE  , wxCMD_LINE_OPTION_HELP      },
  { wxCMD_LINE_SWITCH, "c", "", "", wxCMD_LINE_VAL_NONE                                },
  { wxCMD_LINE_SWITCH, "p", "", "", wxCMD_LINE_VAL_NONE                                },
  { wxCMD_LINE_OPTION, "r", "", "", wxCMD_LINE_VAL_STRING, wxCMD_LINE_OPTION_MANDATORY },
  { wxCMD_LINE_NONE }
};

//**************************************************************************************************

int  main( int argc, char * argv[ ] )
{
  // This function is used in wxBase only and only if an wxApp object isn't created at all. In this
  // case wxInitialize( ) must be called in main( ) before calling any other wxWidgets functions.
  if( ! wxInitialize( argc, argv ) ) exit( EXIT_FAILURE );

  wxCmdLineParser  oCmdLnPsr;
  wxFileName       ofnFileName;
  wxString         osFileName;

  // Setup the command line parser object and process the command line
  oCmdLnPsr.SetDesc( tCmdLnDesc );
  oCmdLnPsr.SetCmdLine( argc, argv );
  if( oCmdLnPsr.Parse( false ) != 0 ) { Usage( argv[0] ); exit( EXIT_FAILURE ); }
  if( oCmdLnPsr.Found( "h" ) )        { Usage( argv[0] ); exit( EXIT_SUCCESS ); }
  oCmdLnPsr.Found( "r", &osFileName );
  ofnFileName = osFileName;
  if( ofnFileName.GetPath( ).IsEmpty( ) ) ofnFileName.SetPath( wxGetenv( "PWD" ) );

  // Create the Config object
  Config  oConfig;

  // Display the utility banner
  std::cout << "\n  Config Class Test Utility"
            << "\n  Version 1.01 (2020-11-01)\n";

  // Configure and create the global configuration object
  if( ! oConfig.bOpen( ofnFileName.GetFullPath( ) ) )
  {
    std::cout << "\nCouldn't open file : " << ofnFileName.GetFullPath( ).mb_str( ) << "\n\n";
    exit( EXIT_FAILURE );
  }

  if( oCmdLnPsr.Found( "c" ) )
  {
    std::cout << "\nClean / rebuild the configuration file : "
              << ( oConfig.bClean( ) ? "Success" : "Failure" ) << "\n\n";
  }
  if( oCmdLnPsr.Found( "p" ) )
  {
    std::cout << "\nPrint oConfig attributes :\n\n";
    oConfig.Print( );
    cout << "\n";
  }

  // This function must be called for each successful call to wxInitialize( ), after this call
  // the library wxWidgets library can no longer be used by this application.
  wxUninitialize( );

  exit( EXIT_SUCCESS );
}

//**************************************************************************************************

void  Usage( char * psAppName )
{
  std::cout << "\nUsage   : " << psAppName << " [-OPTIONS] [FILE]"
            << "\nOptions : -h        : Print usage (this message)"
            << "\n          -r RCFILE : Specify a configuration file"
            << "\n          -c        : Rebuild/clean the configuration file"
            << "\n          -p        : Print configuration object attributes"
            << "\n\n";
}

#endif // TEST_CONFIG

//**************************************************************************************************
